package com.zzsn.event.service.impl;

import cn.hutool.core.net.URLDecoder;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Charsets;
import com.zzsn.event.service.EsDataSearchService;
import com.zzsn.event.util.EsDateUtil;
import com.zzsn.event.vo.SpecialInformationParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.search.join.ScoreMode;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.NestedQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * @Description: 配置关联表
 * @Author: jeecg-boot
 * @Date: 2023-11-20
 * @Version: V1.0
 */
@Slf4j
@Service
public class EsDataSearchServiceServiceImpl implements EsDataSearchService {

    @Override
    public SearchSourceBuilder packageSearQuery(SpecialInformationParam param) {
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        searchSourceBuilder.size(param.getPageSize());
        searchSourceBuilder.from((param.getPageNo() - 1) * param.getPageSize());
        if (StringUtils.isNotEmpty(param.getSearchKey())) {
            //包含检索关键词的时候按照相关度排序
            param.setColumn("score");
        }
        if (param.getColumn().equals("score")) {
            if (param.getOrder().equals("asc")) {
                searchSourceBuilder.sort("_score", SortOrder.ASC);
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
            } else {
                searchSourceBuilder.sort("_score", SortOrder.DESC);
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
            }
        } else if (param.getColumn().equals("common")) {
            if (param.getOrder().equals("asc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
                //第二排序依据为标题倒序
                searchSourceBuilder.sort("title.keyword", SortOrder.ASC);
            } else {
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                //第二排序依据为标题倒序
                searchSourceBuilder.sort("title.keyword", SortOrder.DESC);
            }
        } else if (param.getColumn().equals("topNum")) {
            searchSourceBuilder.sort("topNum", SortOrder.DESC);
        } else {
            if (param.getOrder().equals("asc")) {
                searchSourceBuilder.sort(param.getColumn(), SortOrder.ASC);
            } else {
                searchSourceBuilder.sort(param.getColumn(), SortOrder.DESC);
            }
        }
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        //指定分类标签
        if (StringUtils.isNotEmpty(param.getLabelMarks())) {
            String[] labelMarks = param.getLabelMarks().split(",");
            BoolQueryBuilder boolQuery1 = QueryBuilders.boolQuery();
            for (String labelMark : labelMarks) {
                boolQuery1.should(QueryBuilders.matchPhraseQuery("labels.labelMark", labelMark));
            }
            NestedQueryBuilder nestedQueryBuilder = QueryBuilders
                    .nestedQuery("labels", boolQuery1, ScoreMode.None);
            boolQuery.must(nestedQueryBuilder);
        }
        //封装条件中id为需要排除的数据id
        if (StringUtils.isNotEmpty(param.getId())) {
            boolQuery.mustNot(QueryBuilders.termQuery("id", param.getId()));
        }
        //检索关键词，与数据库配置关键词检索逻辑一致
        if (StringUtils.isNotEmpty(param.getSearchKey())) {
            BoolQueryBuilder builder4 = QueryBuilders.boolQuery();
            for (String keyword : param.getSearchKey().split(" |\\+|,|，|;|；")) {
                if (StringUtils.isEmpty(keyword)) {
                    continue;
                }
                BoolQueryBuilder builder3 = QueryBuilders.boolQuery();
                builder3.should(QueryBuilders.matchQuery("title", keyword).boost(50.0f));
                builder3.should(QueryBuilders.matchQuery("summary", keyword).boost(2.0f));
                builder3.should(QueryBuilders.matchQuery("content", keyword).boost(1.0f));
                if (keyword.startsWith("-")) {
                    builder4.mustNot(builder3);
                } else {
                    builder4.must(builder3);
                }
            }
            boolQuery.must(builder4);
        }
        //标签id
        if (StringUtils.isNotEmpty(param.getSearchLabelIds())) {
            BoolQueryBuilder builder1 = QueryBuilders.boolQuery();
            for (String ids : param.getSearchLabelIds().split(";")) {
                builder1.must(QueryBuilders.nestedQuery("labels", QueryBuilders.termsQuery("labels.relationId", ids.split(",")), ScoreMode.None));
            }
            boolQuery.must(builder1);
        }
        //排除的标签id
        if (StringUtils.isNotEmpty(param.getExcludeLabelIds())) {
            NestedQueryBuilder nestedQueryBuilder = QueryBuilders
                    .nestedQuery("labels", QueryBuilders.termsQuery("labels.relationId", param.getExcludeLabelIds().split(",")), ScoreMode.None);
            boolQuery.mustNot(nestedQueryBuilder);
        }
        //专题id
        if (StringUtils.isNotEmpty(param.getSubjectId())) {
            boolQuery.must(QueryBuilders.termsQuery("subjectId", param.getSubjectId().split(",")));
        }
        if (StringUtils.isNotEmpty(param.getTitle())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("title", param.getTitle()));
        }
        if (StringUtils.isNotEmpty(param.getSummary())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("summary", param.getTitle()));
        }
        if (StringUtils.isNotEmpty(param.getContent())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("content", param.getContent()));
        }
        if (StringUtils.isNotEmpty(param.getOrigin())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("origin", param.getOrigin()));
        }
        if (StringUtils.isNotEmpty(param.getSourceAddress())) {
            boolQuery.must(QueryBuilders.termQuery("sourceAddress.keyword", param.getSourceAddress()));
        }
        //数据状态
        if (StringUtils.isNotEmpty(param.getFindStatus()) && !"ALL".equalsIgnoreCase(param.getFindStatus())) {
            String[] findStatus = param.getFindStatus().split(",");
            boolQuery.must(QueryBuilders.termsQuery("checkStatus", findStatus));
        }
        //发布状态
        if (StringUtils.isNotEmpty(param.getFindPublishStatus()) && !"ALL".equalsIgnoreCase(param.getFindPublishStatus())) {
            List<String> findPublishStatus = new ArrayList<>(Arrays.asList(param.getFindPublishStatus().split(",")));
            if (findPublishStatus.contains("0")) {
                //publishStatus为null的也属于待发布数据
                BoolQueryBuilder builder2 = QueryBuilders.boolQuery();
                builder2.should(QueryBuilders.termsQuery("publishStatus", findPublishStatus));
                builder2.should(QueryBuilders.boolQuery().mustNot(QueryBuilders.existsQuery("publishStatus")));
                boolQuery.must(builder2);
            } else {
                boolQuery.must(QueryBuilders.termsQuery("publishStatus", findPublishStatus));
            }
        }
        if (StringUtils.isNotEmpty(param.getInfoSourceNatureId())) {
            boolQuery.must(QueryBuilders.termsQuery("infoSourceNatureId", param.getInfoSourceNatureId().split(",")));
        }
        //数据类型
        if (StringUtils.isNotEmpty(param.getDataType())) {
            List<String> dataTypes = new ArrayList<>(Arrays.asList(param.getDataType().split(",")));
            if (dataTypes.contains("yqjc")) {
                //包含舆情监测时，舆情预警、竞争情报、口碑监测均不生效
                dataTypes.remove("yqyj");
                dataTypes.remove("jzqb");
                dataTypes.remove("kbjc");
            }
            BoolQueryBuilder builder11 = QueryBuilders.boolQuery();
            for (String dataType : dataTypes) {
                builder11.should(QueryBuilders.termQuery("dataType.keyword", dataType));
            }
            boolQuery.must(builder11);
        }

        //未删除状态查询
        boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));

        //情报时间过滤筛选
        if (StringUtils.isNotBlank(param.getStartTime())) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gte(EsDateUtil.esFieldDateFormat(param.getStartTime())));
        }
        if (StringUtils.isNotBlank(param.getEndTime())) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(param.getEndTime())));
        }
        //本系统发布时间过滤筛选
        if (StringUtils.isNotBlank(param.getPublishStartTime())) {
            boolQuery.filter(QueryBuilders.rangeQuery("updateDate").gte(EsDateUtil.esFieldDateFormat(param.getPublishStartTime())));
        }
        if (StringUtils.isNotBlank(param.getPublishEndTime())) {
            boolQuery.filter(QueryBuilders.rangeQuery("updateDate").lte(EsDateUtil.esFieldDateFormat(param.getPublishEndTime())));
        }
        //高级查询数据处理
        if (StringUtils.isNotEmpty(param.getSuperQueryParams())) {
            try {
                BoolQueryBuilder superBuilder = QueryBuilders.boolQuery();
                String params = URLDecoder.decode(param.getSuperQueryParams(), Charsets.UTF_8);
                JSONArray array = JSON.parseArray(params);
                for (int i = 0; i < array.size(); i++) {
                    JSONObject item = array.getJSONObject(i);
                    BoolQueryBuilder builder2 = QueryBuilders.boolQuery();
                    String type = item.getString("type");
                    String rule = item.getString("rule");
                    String field = item.getString("field");
                    String val = item.getString("val");
                    if ("date".equals(type)) {
                        //日期类型处理
                        if ("eq".equals(rule)) {
                            builder2.must(QueryBuilders.rangeQuery(field).from(val + "T00:00:00").to(val + "T23:59:59"));
                        }
                        if ("ne".equals(rule)) {
                            builder2.mustNot(QueryBuilders.rangeQuery(field).from(val + "T00:00:00").to(val + "T23:59:59"));
                        } else if ("gt".equals(rule)) {
                            builder2.must(QueryBuilders.rangeQuery(field).from((val + "T23:59:59")));
                        } else if ("ge".equals(rule)) {
                            builder2.must(QueryBuilders.rangeQuery(field).from((val + "T00:00:00")));
                        } else if ("lt".equals(rule)) {
                            builder2.must(QueryBuilders.rangeQuery(field).to((val + "T00:00:00")));
                        } else if ("le".equals(rule)) {
                            builder2.must(QueryBuilders.rangeQuery(field).to((val + "T23:59:59")));
                        }
                    } else if ("list".equals(type)) {
                        //列表（数组）类查询
                        if ("eq".equals(rule)) {
                            builder2.must(QueryBuilders.termQuery(field + ".keyword", val));
                        } else if ("ne".equals(rule)) {
                            builder2.mustNot(QueryBuilders.termQuery(field + ".keyword", val));
                        } else if ("in".equals(rule)) {
                            builder2.must(QueryBuilders.termsQuery(field + ".keyword", val.split(" |\\+|,|，|;|；")));
                        } else if ("not_in".equals(rule)) {
                            builder2.mustNot(QueryBuilders.termsQuery(field + ".keyword", val.split(" |\\+|,|，|;|；")));
                        }
                    } else if ("string".equals(type)) {
                        //文本类查询
                        if ("eq".equals(rule)) {
                            builder2.must(QueryBuilders.termQuery(field + ".keyword", val));
                        } else if ("ne".equals(rule)) {
                            builder2.mustNot(QueryBuilders.termQuery(field + ".keyword", val));
                        } else if ("in".equals(rule)) {
                            builder2.must(QueryBuilders.termsQuery(field + ".keyword", val.split(" |\\+|,|，|;|；")));
                        } else if ("not_in".equals(rule)) {
                            builder2.mustNot(QueryBuilders.termsQuery(field + ".keyword", val.split(" |\\+|,|，|;|；")));
                        } else if ("like".equals(rule)) {
                            builder2.must(QueryBuilders.matchPhraseQuery(field, val));
                        } else if ("not_like".equals(rule)) {
                            builder2.mustNot(QueryBuilders.matchPhraseQuery(field, val));
                        }
                    }
                    if ("or".equals(param.getSuperQueryMatchType())) {
                        superBuilder.should(builder2);
                    } else {
                        superBuilder.must(builder2);
                    }
                }
                boolQuery.must(superBuilder);
            } catch (Exception e) {
                log.error("高级查询条件封装失败,e:{},params:{}", e.getMessage(), param.getSuperQueryParams());
            }
        }
        if (param.getHasImg() != null && param.getHasImg() == 1) {
            //包含 img 并且包含 src 的认为包含
            boolQuery.must(QueryBuilders.matchPhraseQuery("contentWithTag", "img"));
            boolQuery.must(QueryBuilders.matchPhraseQuery("contentWithTag", "src"));
        }
        searchSourceBuilder.query(boolQuery);
        String[] include = null;
        String[] exclude = null;
        if (null!=param.sourceInclude) {
            include = param.sourceInclude.split(",");
        }
        //设置排除字段
        if (null!=param.sourceExclude) {
            exclude = param.sourceExclude.split(",");
        }
        //查询指定字段
        searchSourceBuilder.fetchSource(include, exclude);

        return searchSourceBuilder;
    }

}
