package com.zzsn.event.service.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.LabelEntity;
import com.zzsn.event.enums.InfluenceEnum;
import com.zzsn.event.enums.SourceEnum;
import com.zzsn.event.service.EsDataSearchService;
import com.zzsn.event.service.EsStatisticsService;
import com.zzsn.event.service.LabelEntityService;
import com.zzsn.event.util.CalculateUtil;
import com.zzsn.event.vo.CountVO;
import com.zzsn.event.vo.NegativeDataVO;
import com.zzsn.event.vo.SpecialInformationParam;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.BucketOrder;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramInterval;
import org.elasticsearch.search.aggregations.bucket.histogram.Histogram;
import org.elasticsearch.search.aggregations.bucket.histogram.ParsedDateHistogram;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.Sum;
import org.elasticsearch.search.aggregations.metrics.SumAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 舆情信息统计 es查询工具类
 *
 * @author lkg
 * @date 2024/1/19
 */
@Service
public class EsStatisticsServiceImpl implements EsStatisticsService {

    @Resource
    private RestHighLevelClient client;
    @Autowired
    private LabelEntityService labelEntityService;
    @Autowired
    private EsDataSearchService dataSearchService;

    @Override
    public CountVO total(String startTime, String endTime, Integer type) {
        CountVO countVO = new CountVO();
        countVO.setName("舆情信息");
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(null, null, startTime, endTime, null);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        DateHistogramAggregationBuilder aggregation = AggregationBuilders.dateHistogram("group_day")
                .field("publishDate");
        if (type == 1) {
            aggregation.calendarInterval(DateHistogramInterval.HOUR).format("yyyy-MM-dd HH");
        } else if (type == 2) {
            aggregation.calendarInterval(DateHistogramInterval.DAY).format("yyyy-MM-dd");
        }
        searchSourceBuilder.aggregation(aggregation);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            long value = response.getHits().getTotalHits().value;
            countVO.setValue(value);
            List<CountVO> list = new ArrayList<>();
            Aggregations aggregations = response.getAggregations();
            ParsedDateHistogram groupHour = aggregations.get("group_day");
            List<? extends Histogram.Bucket> buckets = groupHour.getBuckets();
            if (CollectionUtils.isNotEmpty(buckets)) {
                for (Histogram.Bucket bucket : buckets) {
                    CountVO vo = new CountVO();
                    vo.setName(bucket.getKeyAsString());
                    vo.setValue(bucket.getDocCount());
                    list.add(vo);
                }
            }
            countVO.setChildren(list);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return countVO;
    }

    @Override
    public Map<String, Object> totalAndMax(String subjectId, String startTime, String endTime) {
        Map<String, Object> map = new HashMap<>();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(subjectId, null, startTime, endTime, null);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        DateHistogramAggregationBuilder aggregation = AggregationBuilders.dateHistogram("group_hour")
                .field("publishDate")
                .calendarInterval(DateHistogramInterval.HOUR)
                .format("yyyy-MM-dd HH")
                .order(BucketOrder.count(false));
        searchSourceBuilder.aggregation(aggregation);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            long value = response.getHits().getTotalHits().value;
            map.put("totalCount", value);
            Aggregations aggregations = response.getAggregations();
            ParsedDateHistogram groupHour = aggregations.get("group_hour");
            List<? extends Histogram.Bucket> buckets = groupHour.getBuckets();
            if (CollectionUtils.isNotEmpty(buckets)) {
                Histogram.Bucket bucket = buckets.get(0);
                long count = bucket.getDocCount();
                map.put("max", count);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return map;
    }


    @Override
    public Map<String, Object> findOne(String subjectId, String labelId, int type, String startTime, String endTime) {
        Map<String, Object> map = new HashMap<>();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(subjectId, labelId, startTime, endTime, type);
        searchSourceBuilder.size(1);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] hits = response.getHits().getHits();
            if (hits.length > 0) {
                SearchHit hit = hits[0];
                map = hit.getSourceAsMap();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return map;
    }


    @Override
    public CountVO orientation(String subjectId, String labelId, String startTime, String endTime, Integer type) {
        CountVO countVO = new CountVO();
        LabelEntity labelEntity = labelEntityService.getById(labelId);
        countVO.setName(labelEntity.getName());
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(subjectId, labelId, startTime, endTime, null);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        DateHistogramAggregationBuilder aggregation = AggregationBuilders.dateHistogram("group_day")
                .field("publishDate");
        if (type == 1) {
            aggregation.calendarInterval(DateHistogramInterval.HOUR).format("yyyy-MM-dd HH");
        } else if (type == 2) {
            aggregation.calendarInterval(DateHistogramInterval.DAY).format("yyyy-MM-dd");
        }
        searchSourceBuilder.aggregation(aggregation);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            long value = response.getHits().getTotalHits().value;
            countVO.setValue(value);
            List<CountVO> list = new ArrayList<>();
            Aggregations aggregations = response.getAggregations();
            ParsedDateHistogram groupHour = aggregations.get("group_day");
            List<? extends Histogram.Bucket> buckets = groupHour.getBuckets();
            if (CollectionUtils.isNotEmpty(buckets)) {
                for (Histogram.Bucket bucket : buckets) {
                    CountVO vo = new CountVO();
                    vo.setName(bucket.getKeyAsString());
                    vo.setValue(bucket.getDocCount());
                    list.add(vo);
                }
            }
            countVO.setChildren(list);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return countVO;
    }


    @Override
    public List<CountVO> source(String subjectId, String startTime, String endTime, Integer type) {
        List<CountVO> list = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(subjectId, null, startTime, endTime, null);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        DateHistogramAggregationBuilder dateAgg = AggregationBuilders.dateHistogram("group_day").field("publishDate");
        if (type == 1) {
            dateAgg.calendarInterval(DateHistogramInterval.HOUR).format("yyyy-MM-dd HH");
        } else if (type == 2) {
            dateAgg.calendarInterval(DateHistogramInterval.DAY).format("yyyy-MM-dd");
        }
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_source")
                .field("infoSourceTypeId.keyword")
                .size(SourceEnum.values().length)
                .subAggregation(dateAgg);
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            long value = response.getHits().getTotalHits().value;
            Aggregations aggregations = response.getAggregations();
            Terms groupSource = aggregations.get("group_source");
            List<? extends Terms.Bucket> sourceBuckets = groupSource.getBuckets();
            for (Terms.Bucket source : sourceBuckets) {
                CountVO countVO = new CountVO();
                String keyAsString = source.getKeyAsString();
                countVO.setName(SourceEnum.getDescription(keyAsString));
                long count = source.getDocCount();
                countVO.setValue(count);
                String percentage = getPercentage(count, value);
                countVO.setPercentage(percentage);
                List<CountVO> children = new ArrayList<>();
                Aggregations sourceAggregations = source.getAggregations();
                ParsedDateHistogram groupDay = sourceAggregations.get("group_day");
                List<? extends Histogram.Bucket> dayBuckets = groupDay.getBuckets();
                for (Histogram.Bucket day : dayBuckets) {
                    CountVO vo = new CountVO();
                    vo.setName(day.getKeyAsString());
                    vo.setValue(day.getDocCount());
                    children.add(vo);
                }
                countVO.setChildren(children);
                list.add(countVO);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }

//    @Override
//    public List<CountVO> region(String subjectId, String startTime, String endTime) {
//        List<SysBaseDomesticRegion> regionList = provinceList();
//        List<CountVO> list = new ArrayList<>();
//        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
//        SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(subjectId, null, startTime, endTime, null, user);
//        searchSourceBuilder.size(0);
//        searchSourceBuilder.trackTotalHits(true);
//        NestedAggregationBuilder nestedAggregationBuilder = AggregationBuilders.nested("labels", "labels")
//                .subAggregation(AggregationBuilders.terms("group_type")
//                        .field("labels.labelMark.keyword")
//                        .subAggregation(AggregationBuilders.terms("group_code")
//                                .field("labels.relationId")
//                                .order(BucketOrder.count(false))
//                                .size(regionList.size())));
//        searchSourceBuilder.aggregation(nestedAggregationBuilder);
//        searchRequest.source(searchSourceBuilder);
//        try {
//            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
//            //获取分组桶
//            Aggregations aggregations = searchResponse.getAggregations();
//            ParsedNested labels = aggregations.get("labels");
//            Aggregations labelsAggregations = labels.getAggregations();
//            //获取按企业信用代码分组集合
//            Terms groupType = labelsAggregations.get("group_type");
//            List<? extends Terms.Bucket> typeBuckets = groupType.getBuckets();
//            Map<String, Long> regionMap = new HashMap<>();
//            for (Terms.Bucket type : typeBuckets) {
//                String key = type.getKeyAsString();
//                if ("region_out".equals(key)) {
//                    Aggregations codeAggregations = type.getAggregations();
//                    Terms groupCode = codeAggregations.get("group_code");
//                    List<? extends Terms.Bucket> codeBuckets = groupCode.getBuckets();
//                    for (Terms.Bucket code : codeBuckets) {
//                        regionMap.put(code.getKeyAsString(), code.getDocCount());
//                    }
//                    break;
//                }
//            }
//            for (SysBaseDomesticRegion region : regionList) {
//                CountVO countVO = new CountVO();
//                String id = region.getId();
//                String name = region.getName();
//                countVO.setName(name);
//                if (regionMap.containsKey(id)) {
//                    Long count = regionMap.get(id);
//                    countVO.setValue(count);
//                } else {
//                    countVO.setValue(0L);
//                }
//                list.add(countVO);
//            }
//            list.sort((o1, o2) -> o2.getValue().compareTo(o1.getValue()));
//            for (int i = 0; i < list.size(); i++) {
//                CountVO countVO = list.get(i);
//                countVO.setOrder(i + 1);
//            }
//        } catch (Exception e) {
//            e.printStackTrace();
//        }
//        return list;
//    }


    @Override
    public List<CountVO> influence(String subjectId, String startTime, String endTime) {
        List<CountVO> list = new ArrayList<>();
        InfluenceEnum[] influenceEnums = InfluenceEnum.values();
        for (InfluenceEnum influenceEnum : influenceEnums) {
            String field = influenceEnum.getField();
            String description = influenceEnum.getDescription();
            SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
            SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(subjectId, null, startTime, endTime, null);
            searchSourceBuilder.size(0);
            searchSourceBuilder.trackTotalHits(true);
            SumAggregationBuilder sumAggregationBuilder = AggregationBuilders.sum("sum_count").field(field);
            searchSourceBuilder.aggregation(sumAggregationBuilder);
            searchRequest.source(searchSourceBuilder);
            try {
                SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
                Aggregations aggregations = response.getAggregations();
                Sum sumCount = aggregations.get("sum_count");
                double value = sumCount.getValue();
                CountVO countVO = new CountVO();
                countVO.setName(description);
                countVO.setValue((long) value);
                list.add(countVO);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return list;
    }

//
//    @Override
//    public List<MediaVO> media(String subjectId, String startTime, String endTime) {
//        List<MediaVO> list = new ArrayList<>();
//        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
//        SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(subjectId, null, startTime, endTime, null, user);
//        searchSourceBuilder.size(0);
//        searchSourceBuilder.trackTotalHits(true);
//        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_nature")
//                .field("infoSourceNatureId.keyword")
//                .subAggregation(AggregationBuilders.terms("group_sid")
//                        .field("sid.keyword")
//                        .order(BucketOrder.count(false))
//                        .size(3))
//                .subAggregation(AggregationBuilders.cardinality("sid_count").field("sid.keyword"));
//        searchSourceBuilder.aggregation(aggregationBuilder);
//        searchRequest.source(searchSourceBuilder);
//        try {
//            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
//            long totalCount = response.getHits().getTotalHits().value;
//            Aggregations aggregations = response.getAggregations();
//            Terms groupNature = aggregations.get("group_nature");
//            List<? extends Terms.Bucket> natureBuckets = groupNature.getBuckets();
//            for (Terms.Bucket nature : natureBuckets) {
//                MediaVO vo = new MediaVO();
//                String natureName = infoSourceService.getNatureName(nature.getKeyAsString());
//                vo.setName(natureName);
//                long count = nature.getDocCount();
//                vo.setValue(count);
//                String percentage = getPercentage(count, totalCount);
//                vo.setPercentage(percentage);
//                List<String> hotList = new ArrayList<>();
//                Aggregations natureAggregations = nature.getAggregations();
//                Terms groupSid = natureAggregations.get("group_sid");
//                List<? extends Terms.Bucket> sidBuckets = groupSid.getBuckets();
//                for (Terms.Bucket sid : sidBuckets) {
//                    String key = sid.getKeyAsString();
//                    hotList.add(key);
//                }
//                List<String> nameList = infoSourceService.infoNameList(hotList);
//                vo.setHotList(nameList);
//                Cardinality sidCount = natureAggregations.get("sid_count");
//                long value = sidCount.getValue();
//                vo.setMediaNum(value);
//                list.add(vo);
//            }
//        } catch (Exception e) {
//            e.printStackTrace();
//        }
//        return list;
//    }

    @Override
    public List<CountVO> origin(String subjectId, String startTime, String endTime) {
        List<CountVO> list = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(subjectId, null, startTime, endTime, null);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_origin")
                .field("origin.keyword")
                .order(BucketOrder.count(false))
                .size(10);
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            long value = response.getHits().getTotalHits().value;
            Aggregations aggregations = response.getAggregations();
            Terms groupSource = aggregations.get("group_origin");
            List<? extends Terms.Bucket> buckets = groupSource.getBuckets();
            if (CollectionUtils.isNotEmpty(buckets)) {
                for (int i = 0; i < buckets.size(); i++) {
                    Terms.Bucket bucket = buckets.get(i);
                    CountVO vo = new CountVO();
                    vo.setOrder(i + 1);
                    vo.setName(bucket.getKeyAsString());
                    long count = bucket.getDocCount();
                    vo.setValue(count);
                    String percentage = getPercentage(count, value);
                    vo.setPercentage(percentage);
                    list.add(vo);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }

    @Override
    public String mainReport(String subjectId) {
        String title = null;
        String mark = getMark(subjectId);
        SearchRequest searchRequest = new SearchRequest(Constants.ES_REPEAT_OLD);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("subjectId", subjectId));
        if (StringUtils.isNotEmpty(mark)) {
            boolQuery.must(QueryBuilders.termQuery("repeatMark", mark));
        }
        boolQuery.must(QueryBuilders.termQuery("saveType", 1));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] hits = response.getHits().getHits();
            SearchHit hit = hits[0];
            title = hit.getSourceAsMap().get("title").toString();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return title;
    }


    @Override
    public Page<NegativeDataVO> labelPageList(String labelId, String startTime, String endTime, Integer pageNo, Integer pageSize) {
        Page<NegativeDataVO> page = new Page<>();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(null, labelId, startTime, endTime, null);
        int offset = (pageNo - 1) * pageSize;
        searchSourceBuilder.from(offset);
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.trackTotalHits(true);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            long count = response.getHits().getTotalHits().value;
            SearchHit[] hits = response.getHits().getHits();
            List<NegativeDataVO> list = new ArrayList<>();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                NegativeDataVO negativeDataVO = JSON.parseObject(sourceAsString, NegativeDataVO.class);
                list.add(negativeDataVO);
            }
            page = new Page<>(pageNo, pageSize, count);
            page.setRecords(list);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return page;
    }

    private String getMark(String subjectId) {
        String mark = null;
        SearchRequest searchRequest = new SearchRequest(Constants.ES_REPEAT_OLD);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("subjectId", subjectId));
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_mark")
                .field("repeatMark")
                .order(BucketOrder.count(false))
                .size(1);
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = response.getAggregations();
            Terms groupSource = aggregations.get("group_mark");
            List<? extends Terms.Bucket> buckets = groupSource.getBuckets();
            if (CollectionUtils.isNotEmpty(buckets)) {
                Terms.Bucket bucket = buckets.get(0);
                mark = bucket.getKeyAsString();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return mark;
    }


    /**
     * 构造查询条件
     *
     * @param subjectId  专题id
     * @param relationId 关联id(可以是标签id，企业信用代码等)
     * @param startTime  开始时间
     * @param endTime    结束时间
     * @param type       1-正序;2-倒序
     * @author lkg
     * @date 2024/1/23
     */
    private SearchSourceBuilder formatSourceBuilder(String subjectId, String relationId, String startTime, String endTime, Integer type) {
        SpecialInformationParam param = new SpecialInformationParam();
        if (StringUtils.isNotEmpty(subjectId)) {
            param.setSubjectId(subjectId);
        } else {
            param.setDataType("yqjc");
        }
        if (StringUtils.isNotEmpty(relationId)) {
            param.setSearchLabelIds(relationId);
        }
        if (StringUtils.isNotBlank(startTime)) {
            param.setStartTime(startTime);
        }
        if (StringUtils.isNotBlank(endTime)) {
            param.setEndTime(endTime);
        }
        param.setColumn("publishDate");
        if (type != null) {
            if (type == 1) {
                param.setOrder("asc");
            } else if (type == 2){
                param.setOrder("desc");
            }
        } else {
            param.setOrder("desc");
        }
        return dataSearchService.packageSearQuery(param);
    }

//    /**
//     * 获取一级地域信息
//     *
//     * @author lkg
//     * @date 2024/1/24
//     */
//    private List<SysBaseDomesticRegion> provinceList() {
//        LambdaQueryWrapper<SysBaseDomesticRegion> queryWrapper = Wrappers.lambdaQuery();
//        queryWrapper.select(SysBaseDomesticRegion::getId, SysBaseDomesticRegion::getName, SysBaseDomesticRegion::getLevel);
//        queryWrapper.eq(SysBaseDomesticRegion::getLevel, 1);
//        return domesticRegionService.list(queryWrapper);
//    }

    /**
     * 获取占比
     *
     * @param count      数量
     * @param totalCount 总数量
     * @author lkg
     * @date 2024/1/25
     */
    private String getPercentage(long count, long totalCount) {
        String divide = CalculateUtil.divide(String.valueOf(count), String.valueOf(totalCount));
        String percentage = "0%";
        if (StringUtils.isNotEmpty(divide)) {
            percentage = CalculateUtil.percentage(Double.parseDouble(divide), false);
        }
        return percentage;
    }

}
