package com.zzsn.event.service;

import com.alibaba.fastjson2.JSON;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import com.zzsn.event.constant.Constants;
import com.zzsn.event.util.EsDateUtil;
import com.zzsn.event.vo.SubjectDataVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.RangeQueryBuilder;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;

/**
 * @author lkg
 * @description: es查询
 * @date 2022/7/7 9:57
 */
@Slf4j
@Service
public class EsService {

    @Autowired
    private RestHighLevelClient client;

    /**
     * 获取专题下的资讯
     *
     * @param subjectId   专题id
     * @param startDate   开始时间
     * @param endDate     结束时间
     * @param flag        0-主条目；null表示所有
     * @param fetchFields 抓取的字段
     * @param sort        1-升序；2-降序
     * @param pageNo      当前页
     * @param pageSize    每页返回条数
     */
    public List<SubjectDataVo> getDataBySubjectId(String subjectId, String startDate, String endDate, String flag, String[] fetchFields, Integer sort, Integer pageNo, Integer pageSize) {
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //按时间排序
        if (sort == 1) {
            searchSourceBuilder.sort("publishDate", SortOrder.ASC);
        } else if (sort == 2) {
            searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        }
        //设置需要获取的字段
//        String[] fetchFields = {"id", "subjectId", "title", "content", "publishDate", "origin", "sourceAddress", "masterEntryId","infoSourceType"};
        searchSourceBuilder.fetchSource(fetchFields, null);
        //设置返回条数
        if (pageNo != null) {
            searchSourceBuilder.from((pageNo - 1) * pageSize);
        }
        if (pageSize == null) {
            pageSize = 50000;
        }
        searchSourceBuilder.size(pageSize);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.matchQuery("subjectId", subjectId));
        if (StringUtils.isNotBlank(startDate) || StringUtils.isNotBlank(endDate)) {
            RangeQueryBuilder rangeQueryBuilder = QueryBuilders.rangeQuery("publishDate");
            if (StringUtils.isNotBlank(startDate)) {
                rangeQueryBuilder.gte(EsDateUtil.esFieldDateFormat(startDate));
            }
            if (StringUtils.isNotBlank(endDate)) {
                rangeQueryBuilder.lte(EsDateUtil.esFieldDateFormat(endDate));
            }
            boolQuery.filter(rangeQueryBuilder);
        }
        //只查询主条目且未删除
        if (StringUtils.isNotEmpty(flag)) {
            boolQuery.mustNot(QueryBuilders.matchQuery("flag", flag));
            //未删除
            boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        }
        return formatData(getEsResult(searchSourceBuilder, boolQuery, subjectId));
    }

    /**
     * 获取专题下的资讯数量
     *
     * @param subjectId 专题id
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @param flag      0-主条目；null表示所有
     */
    public Integer count(String subjectId, String startDate, String endDate, String flag) {
        long count = 0L;
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.matchQuery("subjectId", subjectId));
        if (StringUtils.isNotBlank(startDate) || StringUtils.isNotBlank(endDate)) {
            if (StringUtils.isNotBlank(startDate)) {
                boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gte(EsDateUtil.esFieldDateFormat(startDate)));
            }
            if (StringUtils.isNotBlank(endDate)) {
                boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(endDate)));
            }
        }
        //只查询主条目且未删除
        if (StringUtils.isNotEmpty(flag)) {
            boolQuery.mustNot(QueryBuilders.matchQuery("flag", flag));
            //未删除
            boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        }
        SearchHits searchHits = getEsResult(searchSourceBuilder, boolQuery, subjectId);
        if (searchHits != null) {
            count = searchHits.getTotalHits().value;
        }
        return (int) count;
    }

    /**
     * 根据查询条件分页获取资讯
     *
     * @param subjectId 专题id
     * @param title     标题
     * @param keyword   关键词
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @param pageNo    当前页
     * @param pageSize  页面大小
     * @author lkg
     */
    public IPage<SubjectDataVo> pageList(String subjectId, String title, String keyword, String startDate, String endDate, Integer pageNo, Integer pageSize) {
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.from((pageNo - 1) * pageSize);
        //按时间倒序
        searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        //某专题下的信息
        boolQuery.must(QueryBuilders.matchPhraseQuery("subjectId", subjectId));
        //过滤掉重复数据-flag!=0 表示主条目
        boolQuery.mustNot(QueryBuilders.matchQuery("flag", "0"));
        //未删除
        boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        //标题搜索
        if (StringUtils.isNotEmpty(title)) {
            boolQuery.must(QueryBuilders.matchQuery("title", title));
        }
        /*//标题必须匹配命中关键词
        if (StringUtils.isNotEmpty(keyword)) {
            boolQuery.must(QueryBuilders.matchQuery("title", keyword));
        }*/
        //时间过滤
        if (StringUtils.isNotEmpty(startDate) || StringUtils.isNotEmpty(endDate)) {
            RangeQueryBuilder rangeQueryBuilder = QueryBuilders.rangeQuery("publishDate");
            if (StringUtils.isNotBlank(startDate)) {
                rangeQueryBuilder.gte(EsDateUtil.esFieldDateFormat(startDate));
            }
            if (StringUtils.isNotBlank(endDate)) {
                rangeQueryBuilder.lte(EsDateUtil.esFieldDateFormat(endDate));
            }
            boolQuery.filter(rangeQueryBuilder);
        }
        SearchHits searchHits = getEsResult(searchSourceBuilder, boolQuery, subjectId);
        IPage<SubjectDataVo> page = new Page<>();
        if (searchHits != null) {
            page.setCurrent(pageNo);
            page.setSize(pageSize);
            page.setTotal(searchHits.getTotalHits().value);
            page.setRecords(formatData(searchHits));
        }
        return page;
    }

    /**
     * 根据id获取该条信息的所有重复数据
     *
     * @param id        信息id
     * @param subjectId 专题id
     */
    public List<SubjectDataVo> dataById(String subjectId, String id) {
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置需要获取的字段
        String[] fetchFields = {"id", "subjectId", "title", "publishDate", "origin", "sourceAddress", "masterEntryId", "infoSourceType"};
        searchSourceBuilder.fetchSource(fetchFields, null);
        //设置返回条数
        searchSourceBuilder.size(10000);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("subjectId", subjectId));
        boolQuery.must(QueryBuilders.termQuery("masterEntryId", id));
        boolQuery.mustNot(QueryBuilders.termQuery("id", id));
        return formatData(getEsResult(searchSourceBuilder, boolQuery, subjectId));
    }

    public SubjectDataVo queryInfo(String id) {
        SubjectDataVo subjectDataVo = new SubjectDataVo();
        try {
            SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            searchSourceBuilder.size(1);
            //创建查询对象
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termQuery("id",id));
            searchSourceBuilder.query(boolQuery);
            searchRequest.source(searchSourceBuilder);
            SearchResponse searchResponse;
            try {
                searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
                SearchHits hits = searchResponse.getHits();
                if (hits.getTotalHits().value > 0) {
                    SearchHit hit = hits.getAt(0);
                    String sourceAsString = hit.getSourceAsString();
                    subjectDataVo = JSON.parseObject(sourceAsString, SubjectDataVo.class);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        } catch (Exception e) {
            log.info("获取资讯-{}-详情，查询es库失败", id);
            e.printStackTrace();
        }
        return subjectDataVo;
    }

    public int getRepeatNum(String id) {
        int num = 0;
        SearchRequest searchRequest = new SearchRequest(Constants.COLLECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("repeatId", id));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse;
        try {
            searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            num = (int) searchResponse.getHits().getTotalHits().value;
        } catch (Exception e) {
            log.info("获取资讯-{}-重复数，查询es库失败", id);
            e.printStackTrace();
        }
        return num;
    }

    //格式化数据
    private List<SubjectDataVo> formatData(SearchHits searchHits) {
        List<SubjectDataVo> list = new ArrayList<>();
        if (searchHits != null) {
            SearchHit[] hits = searchHits.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                SubjectDataVo subjectDataVo = JSON.parseObject(sourceAsString, SubjectDataVo.class);
                subjectDataVo.setPublishDate(EsDateUtil.esFieldDateMapping(subjectDataVo.getPublishDate()));
                subjectDataVo.setCreateDate(EsDateUtil.esFieldDateMapping(subjectDataVo.getCreateDate()));
                list.add(subjectDataVo);
            }
        }
        return list;
    }

    //获取es查询结果
    private SearchHits getEsResult(SearchSourceBuilder searchSourceBuilder, BoolQueryBuilder boolQuery, String subjectId) {
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //未删除
//        boolQuery.must(QueryBuilders.matchQuery("deleteFlag",0));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse;
        try {
            searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            return searchResponse.getHits();
        } catch (Exception e) {
            log.info("获取专题-{}-下资讯，查询es库失败", subjectId);
            e.printStackTrace();
        }
        return null;
    }
}
