package com.zzsn.event.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.Event;
import com.zzsn.event.entity.EventTag;
import com.zzsn.event.entity.SubjectInfoSourceMap;
import com.zzsn.event.service.*;
import com.zzsn.event.util.HttpUtil;
import com.zzsn.event.util.ObjectUtil;
import com.zzsn.event.util.tree.Node;
import com.zzsn.event.vo.*;
import com.zzsn.event.xxljob.entity.KeyWords;
import com.zzsn.event.xxljob.service.IXxlJobInfoService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import java.util.*;

/**
 * 事件后台管理
 *
 * @author lkg
 * @date 2024/4/8
 */
@Slf4j
@Api(tags = "事件后台管理")
@RestController
@RequestMapping("/manage")
public class EventManageController {

    @Autowired
    private IEventService eventService;
    @Autowired
    private IEventTagService eventTagService;
    @Autowired
    private LabelEntityService labelEntityService;
    @Autowired
    private IXxlJobInfoService iXxlJobInfoService;
    @Autowired
    private ISubjectInfoSourceMapService subjectInfoSourceMapService;
    @Autowired
    private EventRegionMapService eventRegionMapService;

    @Value(("${serviceProject.url:}"))
    private String SERVICE_PROJECT_URL;

    /**
     * 分页列表-后台管理
     *
     * @param eventName 事件名称
     * @param eventType 事件分类id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param order     排序字段
     * @param orderType 排序方式
     * @param pageNo    偏移量
     * @param pageSize  返回条数
     * @author lkg
     * @date 2024/4/8
     */
    @ApiOperation(value = "事件-分页列表查询", notes = "事件-分页列表查询")
    @GetMapping(value = "/pageList")
    public Result<?> pageList(@RequestParam(name = "eventName", required = false) String eventName,
                              @RequestParam(name = "eventType", required = false) Integer eventType,
                              @RequestParam(name = "startTime", required = false) String startTime,
                              @RequestParam(name = "endTime", required = false) String endTime,
                              @RequestParam(name = "order", required = false) String order,
                              @RequestParam(name = "orderType", defaultValue = "asc") String orderType,
                              @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                              @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        IPage<EventManageVO> pageList = eventService.pageList(eventName, eventType, startTime, endTime, order, orderType, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 地域信息-树型结构
     *
     * @param type 类别(1-国际;2-国内)
     * @author lkg
     * @date 2024/4/10
     */
    @GetMapping("/regionTree")
    public Result<?> regionTree(@RequestParam Integer type) {
        List<Node> nodes = labelEntityService.regionTree(type);
        return Result.OK(nodes);
    }

    /**
     * 2.17 上传icon
     *
     * @return
     */
    @PostMapping(value = "/upload")
    @ResponseBody
    public Result<?> uploadKnowledge(HttpServletRequest request) {
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        MultipartFile file = multipartRequest.getFile("file");// 获取上传文件对象
        String url = eventService.upload(file);
        return Result.OK(url);
    }

    /**
     * 1.2 添加
     *
     * @param eventParam
     * @return
     */
    @ApiOperation(value = "事件-添加", notes = "事件-添加")
    @PostMapping(value = "/add")
    public Result<?> add(@RequestBody AddEventParam eventParam) {
        eventParam.setCreateTime(new Date());
        eventParam.setUpdateTime(new Date());
        Event event = eventService.saveMain(eventParam);
        //插入xxlJob
        iXxlJobInfoService.subjectInsert(event);
        return Result.OK();
    }

    /**
     * 1.3 编辑
     *
     * @param addEventParam
     * @return
     */
    @ApiOperation(value = "事件-编辑", notes = "事件-编辑")
    @PostMapping(value = "/edit")
    public Result<?> edit(@RequestBody AddEventParam addEventParam) {
        addEventParam.setUpdateTime(new Date());
        eventService.updateMain(addEventParam);
        //更新xxljob
        List<String> subjectCodes = new ArrayList<>();
        subjectCodes.add(addEventParam.getEventCode());
        iXxlJobInfoService.keyWordsUpdate(subjectCodes, String.valueOf(addEventParam.getStatus()));
        return Result.OK();
    }

    /**
     * 1.4 通过id删除
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "事件-通过id删除", notes = "事件-通过id删除")
    @GetMapping(value = "/delete")
    public Result<?> delete(@RequestParam(name = "id") String id) {
        Event event = eventService.getById(id);
        iXxlJobInfoService.deleteByInfosourceCode(event.getEventCode());
        eventService.deleteMain(id);
        return Result.OK();
    }

    /**
     * 1.5 批量删除
     *
     * @param ids
     * @return
     */
    @ApiOperation(value = "事件-批量删除", notes = "事件-批量删除")
    @GetMapping(value = "/deleteBatch")
    public Result<?> deleteBatch(@RequestParam(name = "ids", required = true) String ids) {
        this.eventService.removeByIds(Arrays.asList(ids.split(",")));
        return Result.OK();
    }

    /**
     * 1.6 通过id查询
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "事件-通过id查询", notes = "事件-通过id查询")
    @GetMapping(value = "/queryById")
    public Result<?> queryById(@RequestParam(name = "id") String id) {
        Event event = eventService.getById(id);
        String relationEvents = event.getRelationEvents();
        if (null != relationEvents) {
            List<String> split = Arrays.asList(relationEvents.split(","));
            List<Event> list = eventService.list(new LambdaQueryWrapper<Event>().in(Event::getId, split));
            event.setRelatedEventList(list);
        }
        EventTag one = eventTagService.getOne(new LambdaQueryWrapper<EventTag>()
                .eq(EventTag::getEventId, event.getId())
                .last(" limit 1"));
        event.setEventTag(one);
        AddEventParam eventParam = new AddEventParam();
        BeanUtils.copyProperties(event, eventParam);
        List<RegionVO> regionList = eventRegionMapService.regionList(event.getId());
        eventParam.setRegionList(regionList);
        return Result.OK(eventParam);
    }

    /**
     * 1.7 发布
     *
     * @return
     */
    @PostMapping(value = "/publish")
    public Result<?> publish(@RequestBody Event event) {
        Event byId = eventService.getById(event.getId());
        byId.setUpdateTime(new Date());
        byId.setPublishDate(event.getPublishDate());
        byId.setPublishStatus(event.getPublishStatus());
        eventService.updateById(byId);
        return Result.OK();
    }


    /**
     * 2.1 专题信息源绑定
     */
    @PostMapping("/infoSourceBind")
    public Object infoSourceBind(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            String url = SERVICE_PROJECT_URL + "event/infoSource/infoSourceBind";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.2 查询信息源组的绑定列表
     */
    @GetMapping("/bindList")
    public Object bindList(InfoSourceVo infoSourceVo,
                           @RequestParam(name = "ynBind") Integer ynBind,
                           @RequestParam(name = "groupId") String groupId,
                           @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                           @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        String url = SERVICE_PROJECT_URL + "event/infoSource/bindList";
        Map<String, String> params = ObjectUtil.objectToMap(infoSourceVo);
        params.put("ynBind", ynBind.toString());
        params.put("groupId", groupId);
        params.put("pageNo", pageNo.toString());
        params.put("pageSize", pageSize.toString());
        return HttpUtil.doGet(url, params, "utf-8");
    }

    /**
     * 2.3 关键词类别
     */
    @GetMapping("/keywordsType/rootListNoPage")
    public Object keywordsList(@RequestParam(name = "contain", defaultValue = "false") Boolean contain,
                               @RequestParam(name = "subjectId", defaultValue = "0") String subjectId) {
        String url = SERVICE_PROJECT_URL + "event/keywordsType/rootListNoPage";
        Map<String, String> params = new HashMap<>();
        params.put("contain", contain.toString());
        params.put("subjectId", subjectId);
        return HttpUtil.doGet(url, params, "utf-8");
    }

    /**
     * 2.4 专题关键词绑定
     */
    @PostMapping("/keyWordsBind")
    public Object keyWordsBind(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            String url = SERVICE_PROJECT_URL + "event/keyWordsBind";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.5 专题关键词绑定
     */
    @PostMapping("/keyWordsEdit")
    public Object keyWordsEdit(@RequestBody SubjectKeywordsMap subjectKeywordsMap) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectKeywordsMap);
            String url = SERVICE_PROJECT_URL + "event/keyWords/edit";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.6 标签查询
     */
    @PostMapping("/label/treeList")
    public Object labelTreeList(@RequestBody SubjectKeywordsMap subjectKeywordsMap) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectKeywordsMap);
            String url = SERVICE_PROJECT_URL + "event/label/treeList";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.7 专题绑定的搜索引擎列表
     *
     * @return
     */
    @ApiOperation(value = "专题绑定的搜索引擎列表", notes = "专题绑定的搜索引擎列表")
    @GetMapping(value = "/bindSearchEngineList")
    public Object bindSearchEngineList(SearchEnginesVo searchEnginesVo) {
        String url = SERVICE_PROJECT_URL + "event/bindSearchEngineList";
        Map<String, String> params = new HashMap<>();
        params.put("searchEnginesVo", searchEnginesVo.toString());
        params.put("subjectId", searchEnginesVo.getSubjectId());
        if (searchEnginesVo.getType() != null) {
            params.put("type", searchEnginesVo.getType().toString());
        }
        return HttpUtil.doGet(url, params, "utf-8");
    }

    /**
     * 2.8 专题搜索引擎绑定
     *
     * @param subjectPage
     * @return
     */
    @ApiOperation(value = "专题搜索引擎绑定", notes = "专题搜索引擎绑定")
    @PostMapping(value = "/searchEngineBind")
    public Object searchEngineBind(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            String url = SERVICE_PROJECT_URL + "event/searchEngineBind";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.9 专题绑定的信息源组列表
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "专题绑定/排除的信息源组列表", notes = "专题绑定/排除的信息源组列表")
    @GetMapping(value = "/bindInfoSourceList")
    public Object bindInfoSourceList(@RequestParam(name = "id") String id) {
        String url = SERVICE_PROJECT_URL + "event/bindInfoSourceList";
        Map<String, String> params = new HashMap<>();
        params.put("id", id);
        return HttpUtil.doGet(url, params, "utf-8");
    }

    /**
     * 2.10 专题信息源绑定删除
     *
     * @param subjectPage
     * @return
     */
    @ApiOperation(value = "专题信息源绑定删除", notes = "专题信息源绑定删除")
    @PostMapping(value = "/deleteBindInfoSource")
    public Object deleteBindInfoSource(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            ;
            String url = SERVICE_PROJECT_URL + "event/deleteBindInfoSource";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.11 专题与信息源关联编辑
     *
     * @param subjectInfoSourceMap
     * @return
     */

    @ApiOperation(value = "专题与信息源关联表-编辑", notes = "专题与信息源关联表-编辑")
    @PostMapping(value = "/SubjectInfoSourceMap/edit")
    public Result<?> edit(@RequestBody SubjectInfoSourceMap subjectInfoSourceMap) {
        subjectInfoSourceMapService.updateMain(subjectInfoSourceMap);
        return Result.OK("编辑成功!");
    }

    /**
     * 2.12 信息源组类别列表查询
     *
     * @param contain
     * @param subjectId
     * @return
     */
    @ApiOperation(value = "信息源组类别-列表查询", notes = "信息源组类别-列表查询")
    @GetMapping(value = "/groupType/rootListNoPage")
    public Object queryPageListNoPage(@RequestParam(name = "contain", defaultValue = "false") Boolean contain,
                                      @RequestParam(name = "keywords", required = false) String keywords,
                                      @RequestParam(name = "subjectId", defaultValue = "0") String subjectId) {
        String url = SERVICE_PROJECT_URL + "event/groupType/rootListNoPage";
        Map<String, String> params = new HashMap<>();
        params.put("contain", contain.toString());
        params.put("keywords", keywords);
        params.put("subjectId", subjectId);
        return HttpUtil.doGet(url, params, "utf-8");
    }

    /**
     * 2.13 专题绑定的关键词组列表
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "专题绑定的关键词组列表", notes = "专题绑定的关键词组列表")
    @GetMapping(value = "/bindKeyWordsList")
    public Object bindKeyWordsList(@RequestParam(name = "id") String id) {
        String url = SERVICE_PROJECT_URL + "event/bindKeyWordsList";
        Map<String, String> params = new HashMap<>();
        params.put("id", id);
        return HttpUtil.doGet(url, params, "utf-8");
    }

    /**
     * 2.14 删除专题关键词绑定
     *
     * @param subjectPage
     * @return
     */
    @ApiOperation(value = "删除专题关键词绑定", notes = "删除专题关键词绑定")
    @PostMapping(value = "/deleteKeyWordsBind")
    public Object deleteKeyWordsBind(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            String url = SERVICE_PROJECT_URL + "event/deleteKeyWordsBind";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.15 关键词管理-分页列表查询
     *
     * @param keyWords
     * @param pageNo
     * @param pageSize
     * @return
     */
    @ApiOperation(value = "关键词管理-分页列表查询", notes = "关键词管理-分页列表查询")
    @GetMapping(value = "/keyWords/listByTypeId")
    public Object queryPageListByTypeId(KeyWords keyWords,
                                        @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                        @RequestParam(name = "wordsName", required = false) String wordsName,
                                        @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                        @RequestParam(name = "keyWordsTypeId", defaultValue = "0") String keyWordsTypeId,
                                        @RequestParam(name = "search", defaultValue = "false") Boolean search,
                                        @RequestParam(name = "subjectId", defaultValue = "0") String subjectId) {

        String url = SERVICE_PROJECT_URL + "event/keyWords/listByTypeId";
        Map<String, String> params = new HashMap<>();
        keyWords.setWordsName(wordsName);
        params.put("wordsName", wordsName);
        params.put("pageNo", pageNo.toString());
        params.put("pageSize", pageSize.toString());
        params.put("keyWordsTypeId", keyWordsTypeId);
        params.put("subjectId", subjectId);
        params.put("search", search.toString());
        return HttpUtil.doGet(url, params, "utf-8");
    }

    /**
     * 模型信息列表
     *
     * @author lkg
     * @date 2024/4/11
     */
    @GetMapping("/modelList")
    public Result<?> modelList() {
        List<ModelVO> modelVOS = eventService.modelList();
        return Result.OK(modelVOS);
    }

    /**
     * 算法模型信息列表
     *
     * @param subjectId 专题id
     * @param type      类型id
     * @author lkg
     * @date 2024/4/11
     */
    @GetMapping("/algorithmModelList")
    public Object algorithmModelList(@RequestParam String subjectId, @RequestParam Integer type) {
        String url = SERVICE_PROJECT_URL + "event/listNoPage";
        Map<String, String> params = new HashMap<>();
        params.put("subjectId", subjectId);
        params.put("type", type.toString());
        return HttpUtil.doGet(url, params, "utf-8");
    }

    /**
     * 模型绑定
     *
     * @author lkg
     * @date 2024/4/11
     */
    @PostMapping("/modelBind")
    public Object modelBind(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            String url = SERVICE_PROJECT_URL + "event/modelBind";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

}
