package com.zzsn.event.service;

import cn.hutool.core.map.MapUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson2.JSON;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.util.EsDateUtil;
import com.zzsn.event.vo.RepeatHold;
import com.zzsn.event.vo.SubjectDataVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.RangeQueryBuilder;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.BucketOrder;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.collapse.CollapseBuilder;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * @author lkg
 * @description: es查询
 * @date 2022/7/7 9:57
 */
@Slf4j
@Service
public class EsService {

    @Autowired
    private RestHighLevelClient client;


    /**
     * 获取专题下的资讯
     *
     * @param subjectId   专题id
     * @param startDate   开始时间
     * @param endDate     结束时间
     * @param fetchFields 抓取的字段
     */
    public List<SubjectDataVo> getDataBySubjectId(String subjectId, String startDate, String endDate, String[] fetchFields) {
        List<SubjectDataVo> allList = new ArrayList<>();
        AtomicBoolean flag = new AtomicBoolean(true);
        AtomicInteger pageNo = new AtomicInteger(1);
        int size = 30;
        do {
            CompletableFuture<List<SubjectDataVo>> async = CompletableFuture.supplyAsync(() -> {
                SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
                //设置需要获取的字段
                searchSourceBuilder.fetchSource(fetchFields, null);
                //设置返回条数
                searchSourceBuilder.from((pageNo.get() - 1) * size);
                searchSourceBuilder.size(size);
                //创建查询对象
                BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
                boolQuery.must(QueryBuilders.matchQuery("subjectId", subjectId));
                if (StringUtils.isNotBlank(startDate) || StringUtils.isNotBlank(endDate)) {
                    RangeQueryBuilder rangeQueryBuilder = QueryBuilders.rangeQuery("publishDate");
                    if (StringUtils.isNotBlank(startDate)) {
                        rangeQueryBuilder.gte(EsDateUtil.esFieldDateFormat(startDate));
                    }
                    if (StringUtils.isNotBlank(endDate)) {
                        rangeQueryBuilder.lte(EsDateUtil.esFieldDateFormat(endDate));
                    }
                    boolQuery.filter(rangeQueryBuilder);
                }
                List<SubjectDataVo> subjectDataVos = formatData(getEsResult(searchSourceBuilder, boolQuery, subjectId));
                if (subjectDataVos.size() < size) {
                    flag.set(false);
                } else {
                    pageNo.getAndIncrement();
                }
                return subjectDataVos;
            });
            try {
                List<SubjectDataVo> subjectDataVos = async.get();
                allList.addAll(subjectDataVos);
            } catch (Exception e) {
                e.printStackTrace();
            }
        } while (flag.get());
        return allList;
    }

    /**
     * 获取专题下的资讯数量
     *
     * @param subjectId 专题id
     * @param startDate 开始时间
     * @param endDate   结束时间
     */
    public int count(String subjectId, String startDate, String endDate) {
        long count = 0L;
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.matchQuery("subjectId", subjectId));
        if (StringUtils.isNotBlank(startDate) || StringUtils.isNotBlank(endDate)) {
            if (StringUtils.isNotBlank(startDate)) {
                boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gte(EsDateUtil.esFieldDateFormat(startDate)));
            }
            if (StringUtils.isNotBlank(endDate)) {
                boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(endDate)));
            }
        }
        SearchHits searchHits = getEsResult(searchSourceBuilder, boolQuery, subjectId);
        if (searchHits != null) {
            count = searchHits.getTotalHits().value;
        }
        return (int) count;
    }

    /**
     * 根据查询条件分页获取资讯
     *
     * @param subjectId   专题id
     * @param startDate   开始时间
     * @param endDate     结束时间
     * @param fetchFields 抓取的字段
     * @param pageNo      当前页
     * @param pageSize    页面大小
     * @author lkg
     */
    public List<SubjectDataVo> pageList(String subjectId, String startDate, String endDate, String[] fetchFields, Integer sort, Integer pageNo, Integer pageSize) {
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.from((pageNo - 1) * pageSize);
        //按时间排序
        if (sort == 1) {
            searchSourceBuilder.sort("publishDate", SortOrder.ASC);
        } else if (sort == 2) {
            searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        }
        //设置需要获取的字段
        searchSourceBuilder.fetchSource(fetchFields, null);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        //某专题下的信息
        boolQuery.must(QueryBuilders.matchPhraseQuery("subjectId", subjectId));
        //时间过滤
        if (StringUtils.isNotEmpty(startDate) || StringUtils.isNotEmpty(endDate)) {
            RangeQueryBuilder rangeQueryBuilder = QueryBuilders.rangeQuery("publishDate");
            if (StringUtils.isNotBlank(startDate)) {
                rangeQueryBuilder.gte(EsDateUtil.esFieldDateFormat(startDate));
            }
            if (StringUtils.isNotBlank(endDate)) {
                rangeQueryBuilder.lte(EsDateUtil.esFieldDateFormat(endDate));
            }
            boolQuery.filter(rangeQueryBuilder);
        }
        return formatData(getEsResult(searchSourceBuilder, boolQuery, subjectId));
    }

    /**
     * 专题资讯分页列表-门户
     *
     * @param subjectIdList 专题id集合
     * @param searchWord    搜索词
     * @param position      搜索位置(title-标题;content-内容)
     * @param category      匹配度(1-模糊;2-精确)
     * @param column        排序字段
     * @param order         排序方式(asc-正序;desc-倒序)
     * @param pageNo        当前页
     * @param pageSize      返回条数
     * @author lkg
     * @date 2024/4/10
     */
    public IPage<SubjectDataVo> frontListByPage(List<String> subjectIdList, String searchWord, String position, Integer category,
                                                String column, String order, int pageNo, int pageSize) {
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.from((pageNo - 1) * pageSize);
        if (column.equals("score")) {
            if (order.equals("asc")) {
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.ASC));
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
            } else if (order.equals("desc")) {
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.DESC));
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
            }
        } else if (column.equals("publishDate")) {
            if (order.equals("desc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.DESC));
            } else if (order.equals("asc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.ASC));
            }
        }
        String[] fetchFields = new String[]{"id", "subjectId", "title", "summary", "origin", "publishDate", "sourceAddress", "author"};
        searchSourceBuilder.fetchSource(fetchFields, null);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("subjectId.keyword", subjectIdList));
        if (StringUtils.isNotEmpty(searchWord)) {
            if (category == 1) {
                boolQuery.must(QueryBuilders.matchQuery(position, searchWord));
            } else if (category == 2) {
                boolQuery.must(QueryBuilders.matchPhraseQuery(position, searchWord));
            }
        }
        boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        IPage<SubjectDataVo> pageData = new Page<>();
        try {
            List<SubjectDataVo> list = new ArrayList<>();
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            for (SearchHit hit : searchHits) {
                String queryInfo = hit.getSourceAsString();
                SubjectDataVo info = JSONUtil.toBean(queryInfo, SubjectDataVo.class);
                info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
                String index = hit.getIndex();
                info.setIndex(index);
                list.add(info);
            }
            pageData = new Page<>(pageNo, pageSize, searchResponse.getHits().getTotalHits().value);
            pageData.setRecords(list);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return pageData;
    }

    /**
     * 查看资讯详情
     *
     * @param index 索引名称
     * @param id    资讯id
     * @author lkg
     * @date 2024/4/10
     */
    public SubjectDataVo queryInfo(String index, String id) {
        SubjectDataVo subjectDataVo = new SubjectDataVo();
        try {
            SearchRequest searchRequest = new SearchRequest(index);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            searchSourceBuilder.size(1);
            //创建查询对象
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termQuery("id", id));
            searchSourceBuilder.query(boolQuery);
            searchRequest.source(searchSourceBuilder);
            SearchResponse searchResponse;
            try {
                searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
                SearchHits hits = searchResponse.getHits();
                if (hits.getTotalHits().value > 0) {
                    SearchHit hit = hits.getAt(0);
                    String sourceAsString = hit.getSourceAsString();
                    subjectDataVo = JSON.parseObject(sourceAsString, SubjectDataVo.class);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return subjectDataVo;
    }

    /**
     * 根据标题获取相关推荐文章
     *
     * @param id       资讯id
     * @param title    标题
     * @param pageNo   当前页
     * @param pageSize 返回个数
     * @author lkg
     * @date 2024/4/10
     */
    public List<SubjectDataVo> queryRecommendList(String subjectId,String id, String title, Integer pageNo, Integer pageSize) {
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        int offset = (pageNo - 1) * pageSize;
        searchSourceBuilder.from(offset);
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.DESC));
        searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        String[] fetchFields = new String[]{"id", "subjectId", "title", "origin", "publishDate", "sourceAddress"};
        searchSourceBuilder.fetchSource(fetchFields, null);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("subjectId.keyword", subjectId));
        boolQuery.must(QueryBuilders.matchQuery("title", title));
        boolQuery.mustNot(QueryBuilders.termQuery("id", id));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        searchSourceBuilder.collapse(new CollapseBuilder("sourceAddress.keyword"));
        searchSourceBuilder.collapse(new CollapseBuilder("title.keyword"));
        List<SubjectDataVo> list = new ArrayList<>();
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            for (SearchHit hit : searchHits) {
                String index = hit.getIndex();
                String queryInfo = hit.getSourceAsString();
                SubjectDataVo info = com.alibaba.fastjson.JSON.parseObject(queryInfo, SubjectDataVo.class);
                info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
                info.setIndex(index);
                list.add(info);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }


    /**
     * 获取相似文章数
     *
     * @param subjectId     专题id
     * @param articleIdList 资讯id集合
     * @author lkg
     * @date 2024/4/10
     */
    public Map<String, Integer> getSimilarNumber(String subjectId, List<String> articleIdList) {
        Map<String, Integer> map = new HashMap<>();
        Map<String, String> markMap = getMark(subjectId, articleIdList);
        if (MapUtil.isNotEmpty(markMap)) {
            List<String> markList = new ArrayList<>(markMap.keySet());
            SearchRequest searchRequest = new SearchRequest(Constants.ES_REPEAT_OLD);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            //创建查询对象
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termsQuery("repeatMark", markList));
            searchSourceBuilder.size(0);
            searchSourceBuilder.trackTotalHits(true);
            TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_mark")
                    .field("repeatMark")
                    .order(BucketOrder.count(false))
                    .size(articleIdList.size());
            searchSourceBuilder.aggregation(aggregationBuilder);
            searchSourceBuilder.query(boolQuery);
            searchRequest.source(searchSourceBuilder);
            try {
                SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
                Aggregations aggregations = response.getAggregations();
                Terms groupSource = aggregations.get("group_mark");
                List<? extends Terms.Bucket> buckets = groupSource.getBuckets();
                if (CollectionUtils.isNotEmpty(buckets)) {
                    for (Terms.Bucket bucket : buckets) {
                        map.put(markMap.get(bucket.getKeyAsString()), (int) bucket.getDocCount());
                    }

                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return map;
    }

    /**
     * 获取资讯所在蔟(蔟中存储重复数据信息)
     *
     * @param subjectId     专题id
     * @param articleIdList 资讯id集合
     * @author lkg
     * @date 2024/4/10
     */
    private Map<String, String> getMark(String subjectId, List<String> articleIdList) {
        Map<String, String> map = new HashMap<>();
        if (CollectionUtils.isNotEmpty(articleIdList)) {
            List<String> idList = new ArrayList<>();
            for (String articleId : articleIdList) {
                idList.add(articleId.replace(subjectId, ""));
            }
            SearchRequest searchRequest = new SearchRequest(Constants.ES_REPEAT_OLD);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            //创建查询对象
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termsQuery("articleId", idList));
            boolQuery.must(QueryBuilders.termQuery("subjectId", subjectId));
            searchSourceBuilder.query(boolQuery);
            searchRequest.source(searchSourceBuilder);
            try {
                SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
                SearchHit[] hits = response.getHits().getHits();
                if (hits != null && hits.length != 0) {
                    for (SearchHit hit : hits) {
                        String queryInfo = hit.getSourceAsString();
                        RepeatHold info = JSONUtil.toBean(queryInfo, RepeatHold.class);
                        map.put(info.getRepeatMark(), info.getSubjectId() + info.getArticleId());
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return map;
    }


    /**
     * 按标题匹配数据并按来源分组
     *
     * @param title       标题
     * @param publishDate 发布时间
     * @author lkg
     * @date 2024/4/11
     */
    public List<String> groupByOrigin(String title, String publishDate) {
        List<String> originList = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.COLLECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //只返回分组聚合结果，不返回具体数据
        searchSourceBuilder.size(0);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        String[] arr = new String[]{"title"};
        boolQuery.must(QueryBuilders.multiMatchQuery(title, arr));
        boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gt(EsDateUtil.esFieldDateFormat(publishDate)));
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_origin")
                .field("origin.keyword")
                .size(20)
                .order(BucketOrder.count(false));
        searchSourceBuilder.query(boolQuery);
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = searchResponse.getAggregations();
            Terms groupOrigin = aggregations.get("group_origin");
            List<? extends Terms.Bucket> buckets = groupOrigin.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                String origin = bucket.getKeyAsString();
                originList.add(origin);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return originList;
    }

    /**
     * 报告导出的资讯信息列表
     *
     * @param subjectIdList 专题id集合
     * @param searchWord    搜索词
     * @param position      搜索位置(title-标题;content-内容)
     * @param category      匹配度(1-模糊;2-精确)
     * @param articleIdList 资讯id集合
     * @param column        排序字段
     * @param order         排序方式(asc-正序;desc-倒序)
     * @param type         导出方式(1-摘要;2-正文)
     * @param pageSize      返回条数
     * @author lkg
     * @date 2024/4/11
     */
    public List<SubjectDataVo> exportDataList(List<String> subjectIdList, String searchWord, String position, Integer category,
                                              List<String> articleIdList, String column, String order,Integer type, Integer pageSize) {
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        if (CollectionUtils.isNotEmpty(articleIdList)) {
            pageSize = articleIdList.size();
        }
        //设置分页参数
        searchSourceBuilder.size(pageSize);
        if (column.equals("score")) {
            if (order.equals("asc")) {
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.ASC));
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
            } else if (order.equals("desc")) {
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.DESC));
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
            }
        } else if (column.equals("publishDate")) {
            if (order.equals("desc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.DESC));
            } else if (order.equals("asc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.ASC));
            }
        }
        String[] fetchFields = new String[]{"id", "title", "summary", "origin", "publishDate", "sourceAddress"};
        if (type == 2) {
            fetchFields = new String[]{"id", "title", "summary", "content", "origin", "publishDate", "sourceAddress"};
        }
        searchSourceBuilder.fetchSource(fetchFields, null);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("subjectId.keyword", subjectIdList));
        if (CollectionUtils.isNotEmpty(articleIdList)) {
            boolQuery.must(QueryBuilders.termsQuery("id", articleIdList));
        }
        if (StringUtils.isNotEmpty(searchWord)) {
            if (category == 1) {
                boolQuery.must(QueryBuilders.matchQuery(position, searchWord));
            } else if (category == 2) {
                boolQuery.must(QueryBuilders.matchPhraseQuery(position, searchWord));
            }
        }
        boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        List<SubjectDataVo> list = new ArrayList<>();
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            for (SearchHit hit : searchHits) {
                String queryInfo = hit.getSourceAsString();
                SubjectDataVo info = JSONUtil.toBean(queryInfo, SubjectDataVo.class);
                info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
                list.add(info);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }


    //格式化数据
    private List<SubjectDataVo> formatData(SearchHits searchHits) {
        List<SubjectDataVo> list = new ArrayList<>();
        if (searchHits != null) {
            SearchHit[] hits = searchHits.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                SubjectDataVo subjectDataVo = JSON.parseObject(sourceAsString, SubjectDataVo.class);
                subjectDataVo.setPublishDate(EsDateUtil.esFieldDateMapping(subjectDataVo.getPublishDate()));
                list.add(subjectDataVo);
            }
        }
        return list;
    }

    //获取es查询结果
    private SearchHits getEsResult(SearchSourceBuilder searchSourceBuilder, BoolQueryBuilder boolQuery, String subjectId) {
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //未删除
        boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse;
        try {
            searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            return searchResponse.getHits();
        } catch (Exception e) {
            log.info("获取专题-{}-下资讯，查询es库失败", subjectId);
            e.printStackTrace();
        }
        return null;
    }
}
