package com.zzsn.event.task;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.alibaba.fastjson2.JSONWriter;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.SubjectAnalysis;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.AnalysisService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.service.SubjectAnalysisService;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.util.RedisUtil;
import com.zzsn.event.util.SimilarityUtil;
import com.zzsn.event.vo.*;
import com.zzsn.event.vo.es.SpecialInformation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.CompletableFuture;

/**
 * @author lkg
 * @description: 专题分析定时任务
 * @date 2022/7/8 11:58
 */
@Slf4j
@Component
public class AnalysisTask {

    @Autowired
    private IEventService eventService;
    @Autowired
    private EsService esService;
    @Resource
    private KafkaTemplate<String, String> kafkaTemplate;
    @Autowired
    private RedisUtil redisUtil;
    @Autowired
    private AnalysisService analysisService;
    @Autowired
    private SubjectAnalysisService subjectAnalysisService;

    @Value("${scheduling.yjzxEnable:false}")
    Boolean yjzxEnable;


    /**
     * 补充事件当天的脉络
     * 每3小时执行一次
     *
     * @author lkg
     * @date 2025/7/9
     */
    @Scheduled(cron = "0 0 0/3 * * ?")
    public void eventContext() {
        Date today = new Date();
        Date disableDate = DateUtil.addDate(today, -1);
        List<SubjectKafkaVo> subjectList = eventService.progressList(disableDate);
        InfoDataSearchCondition searchCondition = new InfoDataSearchCondition();
        searchCondition.setCategory(2);
        String startTime = DateUtil.format(today, "yyyy-MM-dd 00:00:00");
        searchCondition.setStartTime(startTime);
        String endTime = DateUtil.format(today, "yyyy-MM-dd 23:59:59");
        searchCondition.setEndTime(endTime);
        String[] fetchFields = new String[]{"id", "title", "origin", "publishDate", "sourceAddress"};
        searchCondition.setFetchFields(fetchFields);
        for (SubjectKafkaVo subjectKafkaVo : subjectList) {
            String eventId = subjectKafkaVo.getId();
            searchCondition.setSubjectId(eventId);
            int pageNo = 1;
            int size = 100;
            searchCondition.setPageSize(size);
            boolean flag = true;
            List<EventContextVO> dataList = new ArrayList<>();
            do {
                searchCondition.setPageNo(pageNo);
                List<SpecialInformation> informationList = esService.informationList(searchCondition);
                if (CollectionUtils.isNotEmpty(informationList)) {
                    informationList.forEach(e -> {
                        double similarity = SimilarityUtil.levenshteinSimilarity(subjectKafkaVo.getSubjectName(), e.getTitle());
                        if (similarity > 0) {
                            EventContextVO eventContextVO = new EventContextVO();
                            BeanUtils.copyProperties(e, eventContextVO);
                            eventContextVO.setDistance(similarity);
                            dataList.add(eventContextVO);
                        }
                    });
                    if (informationList.size() < size) {
                        flag = false;
                    } else {
                        pageNo++;
                    }
                } else {
                    flag = false;
                }
            } while (flag);
            if (CollectionUtils.isEmpty(dataList)) {
                continue;
            }
            List<EventContextVO> list = supplyMain(dataList);
            LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectAnalysis::getSubjectId, eventId)
                    .eq(SubjectAnalysis::getCategory, 4)
                    .between(SubjectAnalysis::getPublishDate, startTime, endTime);
            int count = subjectAnalysisService.count(queryWrapper);
            if (count > 0) {
                subjectAnalysisService.remove(queryWrapper);
            }
            List<SubjectAnalysis> analysisList = new ArrayList<>();
            for (EventContextVO eventContextVO : list) {
                SubjectAnalysis subjectAnalysis = new SubjectAnalysis();
                subjectAnalysis.setSubjectId(eventId);
                subjectAnalysis.setDataId(eventContextVO.getId());
                subjectAnalysis.setTitle(eventContextVO.getTitle());
                subjectAnalysis.setOrigin(eventContextVO.getOrigin());
                subjectAnalysis.setPublishDate(DateUtil.stringToDate(eventContextVO.getPublishDate(), "yyyy-MM-dd HH:mm:ss"));
                subjectAnalysis.setSourceAddress(eventContextVO.getSourceAddress());
                subjectAnalysis.setIsMain(eventContextVO.getIsMain() ? 1 : 0);
                subjectAnalysis.setCategory(4);
                subjectAnalysis.setAnalysisDate(new Date());
                analysisList.add(subjectAnalysis);
            }
            subjectAnalysisService.saveBatch(analysisList);
        }
    }

    //补充是否主要资讯标识以及排序后截取
    private List<EventContextVO> supplyMain(List<EventContextVO> list) {
        EventContextVO dataVo = null;
        double minDistance = Double.MAX_VALUE;
        for (EventContextVO eventContextVO : list) {
            Double similarity = eventContextVO.getDistance();
            if (similarity == 0) {
                continue;
            }
            if (similarity == 1.0) {
                dataVo = eventContextVO;
                break;
            } else if (similarity < minDistance) {
                minDistance = similarity;
                dataVo = eventContextVO;
            }
        }
        if (dataVo == null) {
            list.get(0).setIsMain(true);
        } else {
            for (EventContextVO eventContextVO : list) {
                if (eventContextVO.getId().equals(dataVo.getId())) {
                    eventContextVO.setIsMain(true);
                    break;
                }
            }
        }
        //先按是否是主事件排序，再按相似度算法(编辑距离)返回值正序排序，最后按时间倒序排序
        list.sort(Comparator.comparing(EventContextVO::getIsMain, Comparator.reverseOrder())
                .thenComparing((o1, o2) -> {
                    Double distance1 = o1.getDistance();
                    Double distance2 = o2.getDistance();
                    if (distance1 == 0 && distance2 == 0) {
                        return 0;
                    }
                    if (distance1 == 0) {
                        return 1;
                    }
                    if (distance2 == 0) {
                        return -1;
                    }
                    return distance1.compareTo(distance2);
                })
                .thenComparing(EventContextVO::getPublishDate, Comparator.reverseOrder()));
        if (list.size() > 3) {
            list = list.subList(0, 3);
        }
        return list;
    }


    /**
     * 每天凌晨0点20分执行一次
     * 发送 伪事件脉络 所需信息到kafka对应的topic
     */
    @Scheduled(cron = "0 20 0 * * ?")
    public void eventContext_fake() {
        if (yjzxEnable) {
            //研究中心不需要此任务
            return;
        }
        Date today = new Date();
        Date disableDate = DateUtil.addDate(today, -1);
        List<SubjectKafkaVo> fakeEventIdList = new ArrayList<>();
        List<SubjectKafkaVo> subjectList = eventService.progressList(disableDate);
        subjectList.forEach(e -> {
            LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectAnalysis::getSubjectId, e.getId()).eq(SubjectAnalysis::getCategory, 2);
            int count = subjectAnalysisService.count(queryWrapper);
            if (count < Constants.FAKE_NUM) {
                fakeEventIdList.add(e);
            }
        });
        for (SubjectKafkaVo subjectKafkaVo : fakeEventIdList) {
            List<KafkaDataVo> kafkaDataVoList = new ArrayList<>();
            String subjectId = subjectKafkaVo.getId();
            String keyWord = subjectKafkaVo.getKeyWord();
            List<SubjectDataVo> dataList = esService.getDataBySubjectId(subjectId, null, null, Constants.FETCH_FIELDS_DATA);
            format(subjectId, kafkaDataVoList, dataList);
            splitSend(Constants.FAKE_EVENT_CONTEXT_SEND_TOPIC, subjectId, kafkaDataVoList, keyWord);
        }
    }

    private void format(String subjectId, List<KafkaDataVo> kafkaDataVoList, List<SubjectDataVo> dataList) {
        dataList.forEach(e -> {
            String dataId = e.getId();
            KafkaDataVo kafkaDataVo = new KafkaDataVo();
            BeanUtils.copyProperties(e, kafkaDataVo);
            List<String> idList = new ArrayList<>();
            idList.add(dataId);
            Map<String, Integer> similarNumber = esService.getSimilarNumber(subjectId, idList);
            Integer count = similarNumber.get(dataId);
            if (count == null) {
                count = 0;
            }
            kafkaDataVo.setRepeatNum(count);
            kafkaDataVoList.add(kafkaDataVo);
        });
    }

    //防止数据量太大，超过kafka的最大值，所以分批发送，一次20条数据
    private void splitSend(String topic, String subjectId, List<KafkaDataVo> list, String keyWord) {
        List<List<KafkaDataVo>> partition = ListUtils.partition(list, 20);
        partition.forEach(e -> {
            Map<String, Object> map = new HashMap<>();
            map.put("keyword", keyWord);
            map.put("data", e);
            kafkaTemplate.send(topic, subjectId, JSON.toJSONString(map, JSONWriter.Feature.WriteMapNullValue));
        });
    }

    /**
     * 定时生成传播路径
     * 每天凌晨0点10分执行一次
     */
    @Scheduled(cron = "0 0 1 * * ?")
    public void propagationPath() {
        Date today = new Date();
        Date deadlineDate = DateUtil.addDate(today, -1);
        List<SubjectKafkaVo> subjects = eventService.eventSubjectList();
        for (SubjectKafkaVo subject : subjects) {
            CompletableFuture.runAsync(() -> {
                String subjectId = subject.getId();
                int count = esService.count(subjectId, null, null);
                if (count > 0) {
                    String key = Constants.SUBJECT_ANALYSIS_PRE + Constants.PROPAGATION_KEY + subjectId;
                    Date timeDisable = subject.getTimeDisable();
                    //已经结束的事件专题，永久缓存
                    if (timeDisable != null && deadlineDate.compareTo(timeDisable) > 0) {
                        Object cacheObject = redisUtil.get(key);
                        if (cacheObject == null) {
                            PropagationPathVo pathVo = analysisService.propagationPath(subjectId);
                            if (ObjectUtils.isNotEmpty(pathVo)) {
                                redisUtil.set(key, pathVo);
                                log.info("专题-{},传播路径数据【永久】缓存成功!", subject.getSubjectName());
                            }
                        }
                    } else {//已经结束的事件专题，缓存有效期一天
                        PropagationPathVo pathVo = analysisService.propagationPath(subjectId);
                        if (ObjectUtils.isNotEmpty(pathVo)) {
                            redisUtil.set(key, pathVo, 3600 * 24);
                            log.info("专题-{},传播路径数据缓存成功!", subject.getSubjectName());
                        }
                    }
                }
            });
        }
    }
}
