package com.zzsn.event.es;

import cn.hutool.core.map.MapUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.ClbChannelLabelMap;
import com.zzsn.event.entity.CollectionMap;
import com.zzsn.event.entity.SubjectAnalysis;
import com.zzsn.event.entity.SysBaseChannel;
import com.zzsn.event.service.*;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.util.EsDateUtil;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.search.join.ScoreMode;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.*;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.BucketOrder;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramInterval;
import org.elasticsearch.search.aggregations.bucket.histogram.Histogram;
import org.elasticsearch.search.aggregations.bucket.histogram.ParsedDateHistogram;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.collapse.CollapseBuilder;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * @author lkg
 * @description: es查询
 * @date 2022/7/7 9:57
 */
@Slf4j
@Service
public class EsService {

    @Resource
    private RestHighLevelClient client;
    @Autowired
    private ISubjectTypeService iSubjectTypeService;
    @Autowired
    private IEventService eventService;
    @Autowired
    private ISubjectInfoSourceMapService iSubjectInfoSourceMapService;
    @Autowired
    private ICollectionMapService collectionMapService;
    @Autowired
    private ISysBaseChannelService channelService;
    @Autowired
    private IClbChannelLabelMapService channelLabelMapService;

    private final static String[] EVENT_ANALYSIS_ORIGIN = new String[]{"CRI国际在线", "world.people", "安徽国企网", "半月谈", "参考消息", "参考消息电子版", "参考消息网", "参考消息智库", "电池中国网", "法制网", "光明经济网", "光明网", "国际在线", "国家工业信息安全发展研究中心", "国企网", "国研网", "环球网", "价值中国网", "经济参考报", "经济日报", "科技日报", "理论网", "企业观察网", "求是网", "人民安全网", "人民报（Nhan Dan Báo）", "人民创投", "人民法治网", "人民交通网", "人民论坛网", "人民日报海外版", "人民数字联播网", "人民网", "人民雄安网", "人民资讯网", "手机央广网", "习近平系列讲话数据库", "新华报业网", "新华网", "央广网", "央视网", "央视网经济", "一带一路网", "正义网", "证券日报", "证券日报网", "中国安全生产网", "中国财经报", "中国发展门户网", "中国发展网", "中国改革报", "中国工商报", "中国工商报网", "中国工业报", "中国工业报社", "中国工业新闻网（中国工业报）电工频道", "中国经济导报", "中国经济导报网", "中国经济网", "中国经济网航空频道", "中国经济网理论评论频道", "中国经济新闻网", "中国经济信息网", "中国警察网", "中国军网", "中国农网", "中国青年报", "中国青年网", "中国台湾网", "中国网", "中国文明网", "中国西藏网", "中宏网", "中华工商网", "中华环境保护基金会", "中青在线", "中央人民广播电台", "观察者网", "澎湃新闻", "凤凰网", "每日经济新闻", "新京报"};

    /**
     * 获取专题下的资讯
     *
     * @param subjectId   专题id
     * @param startDate   开始时间
     * @param endDate     结束时间
     * @param fetchFields 抓取的字段
     */
    public List<SubjectDataVo> getDataBySubjectId(String subjectId, String startDate, String endDate, String[] fetchFields) {
        List<SubjectDataVo> allList = new ArrayList<>();
        AtomicBoolean flag = new AtomicBoolean(true);
        AtomicInteger pageNo = new AtomicInteger(1);
        int size = 300;
        do {
            CompletableFuture<List<SubjectDataVo>> async = CompletableFuture.supplyAsync(() -> {
                SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
                //设置需要获取的字段
                searchSourceBuilder.fetchSource(fetchFields, null);
                //设置返回条数
                searchSourceBuilder.from((pageNo.get() - 1) * size);
                searchSourceBuilder.size(size);
                //创建查询对象
                BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
                boolQuery.must(QueryBuilders.matchQuery("subjectId", subjectId));
                if (StringUtils.isNotBlank(startDate) || StringUtils.isNotBlank(endDate)) {
                    RangeQueryBuilder rangeQueryBuilder = QueryBuilders.rangeQuery("publishDate");
                    if (StringUtils.isNotBlank(startDate)) {
                        rangeQueryBuilder.gte(EsDateUtil.esFieldDateFormat(startDate));
                    }
                    if (StringUtils.isNotBlank(endDate)) {
                        rangeQueryBuilder.lte(EsDateUtil.esFieldDateFormat(endDate));
                    }
                    boolQuery.filter(rangeQueryBuilder);
                }
                boolQuery.mustNot(QueryBuilders.termQuery("origin.keyword", ""));
                //未删除状态查询
                boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
                List<SubjectDataVo> subjectDataVos = formatData(getEsResult(searchSourceBuilder, boolQuery, subjectId));
                if (subjectDataVos.size() < size) {
                    flag.set(false);
                } else {
                    pageNo.getAndIncrement();
                }
                log.info("本次循环-{}，数据量为-{}", pageNo, subjectDataVos.size());
                return subjectDataVos;
            });
            try {
                List<SubjectDataVo> subjectDataVos = async.get();
                allList.addAll(subjectDataVos);
            } catch (Exception e) {
                e.printStackTrace();
            }
        } while (flag.get());
        return allList;
    }

    /**
     * 获取专题下的资讯数量
     *
     * @param subjectId 专题id
     * @param startDate 开始时间
     * @param endDate   结束时间
     */
    public int count(String subjectId, String startDate, String endDate) {
        long count = 0L;
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("subjectId.keyword", subjectId));
        if (StringUtils.isNotBlank(startDate) || StringUtils.isNotBlank(endDate)) {
            if (StringUtils.isNotBlank(startDate)) {
                boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gte(EsDateUtil.esFieldDateFormat(startDate)));
            }
            if (StringUtils.isNotBlank(endDate)) {
                boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(endDate)));
            }
        }
        SearchHits searchHits = getEsResult(searchSourceBuilder, boolQuery, subjectId);
        if (searchHits != null) {
            count = searchHits.getTotalHits().value;
        }
        return (int) count;
    }

    /**
     * 根据查询条件分页获取资讯
     *
     * @param subjectId   专题id
     * @param startDate   开始时间
     * @param endDate     结束时间
     * @param fetchFields 抓取的字段
     * @param pageNo      当前页
     * @param pageSize    页面大小
     * @author lkg
     */
    public List<SubjectDataVo> pageList(String subjectId, String startDate, String endDate, String[] fetchFields, Integer sort, Integer pageNo, Integer pageSize) {
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.from((pageNo - 1) * pageSize);
        //按时间排序
        if (sort == 1) {
            searchSourceBuilder.sort("publishDate", SortOrder.ASC);
        } else if (sort == 2) {
            searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        }
        //设置需要获取的字段
        searchSourceBuilder.fetchSource(fetchFields, null);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        //某专题下的信息
        boolQuery.must(QueryBuilders.matchPhraseQuery("subjectId", subjectId));
        //时间过滤
        if (StringUtils.isNotEmpty(startDate) || StringUtils.isNotEmpty(endDate)) {
            RangeQueryBuilder rangeQueryBuilder = QueryBuilders.rangeQuery("publishDate");
            if (StringUtils.isNotBlank(startDate)) {
                rangeQueryBuilder.gte(EsDateUtil.esFieldDateFormat(startDate));
            }
            if (StringUtils.isNotBlank(endDate)) {
                rangeQueryBuilder.lte(EsDateUtil.esFieldDateFormat(endDate));
            }
            boolQuery.filter(rangeQueryBuilder);
        }
        return formatData(getEsResult(searchSourceBuilder, boolQuery, subjectId));
    }

    /**
     * 专题资讯分页列表-门户
     *
     * @param subjectIdList 专题id集合
     * @param searchWord    搜索词
     * @param position      搜索位置(title-标题;content-内容)
     * @param category      匹配度(1-模糊;2-精确)
     * @param column        排序字段
     * @param order         排序方式(asc-正序;desc-倒序)
     * @param pageNo        当前页
     * @param pageSize      返回条数
     * @author lkg
     * @date 2024/4/10
     */
    public IPage<SubjectDataVo> frontListByPage(List<String> subjectIdList, String searchWord, String position, Integer category,
                                                String labelId, String column, String order, int pageNo, int pageSize) {
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.from((pageNo - 1) * pageSize);
        if (column.equals("score")) {
            if (order.equals("asc")) {
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.ASC));
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
            } else if (order.equals("desc")) {
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.DESC));
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
            }
        } else if (column.equals("publishDate")) {
            if (order.equals("desc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.DESC));
            } else if (order.equals("asc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.ASC));
            }
        }
        String[] fetchFields = new String[]{"id", "subjectId", "title", "summary", "origin", "publishDate", "sourceAddress", "author", "labels"};
        searchSourceBuilder.fetchSource(fetchFields, null);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("subjectId.keyword", subjectIdList));
        if (StringUtils.isNotBlank(searchWord)) {
            if (category == 1) {
                boolQuery.must(QueryBuilders.matchQuery(position, searchWord));
            } else if (category == 2) {
                boolQuery.must(QueryBuilders.matchPhraseQuery(position, searchWord));
            }
        }
        if (StringUtils.isNotBlank(labelId)) {
            boolQuery.must(QueryBuilders.nestedQuery("labels", QueryBuilders.termQuery("labels.relationId", labelId), ScoreMode.None));
        }
        boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        IPage<SubjectDataVo> pageData = new Page<>();
        try {
            List<SubjectDataVo> list = new ArrayList<>();
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            for (SearchHit hit : searchHits) {
                String queryInfo = hit.getSourceAsString();
                SubjectDataVo info = JSONUtil.toBean(queryInfo, SubjectDataVo.class);
                info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
                String index = hit.getIndex();
                info.setIndex(index);
                list.add(info);
            }
            pageData = new Page<>(pageNo, pageSize, searchResponse.getHits().getTotalHits().value);
            pageData.setRecords(list);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return pageData;
    }

    /**
     * 查看资讯详情
     *
     * @param index 索引名称
     * @param id    资讯id
     * @author lkg
     * @date 2024/4/10
     */
    public SubjectDataVo queryInfo(String index, String id) {
        SubjectDataVo subjectDataVo = new SubjectDataVo();
        try {
            SearchRequest searchRequest = new SearchRequest(index);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            searchSourceBuilder.size(1);
            //创建查询对象
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termQuery("id", id));
            searchSourceBuilder.query(boolQuery);
            searchRequest.source(searchSourceBuilder);
            SearchResponse searchResponse;
            try {
                searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
                SearchHits hits = searchResponse.getHits();
                if (hits.getTotalHits().value > 0) {
                    SearchHit hit = hits.getAt(0);
                    String sourceAsString = hit.getSourceAsString();
                    subjectDataVo = JSON.parseObject(sourceAsString, SubjectDataVo.class);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return subjectDataVo;
    }

    /**
     * 根据标题获取相关推荐文章
     *
     * @param id       资讯id
     * @param title    标题
     * @param pageNo   当前页
     * @param pageSize 返回个数
     * @author lkg
     * @date 2024/4/10
     */
    public List<SubjectDataVo> queryRecommendList(String subjectId, String id, String title, Integer pageNo, Integer pageSize) {
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        int offset = (pageNo - 1) * pageSize;
        searchSourceBuilder.from(offset);
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.DESC));
        searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        String[] fetchFields = new String[]{"id", "subjectId", "title", "origin", "publishDate", "sourceAddress"};
        searchSourceBuilder.fetchSource(fetchFields, null);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("subjectId.keyword", subjectId));
        boolQuery.must(QueryBuilders.matchQuery("title", title));
        boolQuery.mustNot(QueryBuilders.termQuery("id", id));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        searchSourceBuilder.collapse(new CollapseBuilder("sourceAddress.keyword"));
        searchSourceBuilder.collapse(new CollapseBuilder("title.keyword"));
        List<SubjectDataVo> list = new ArrayList<>();
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            for (SearchHit hit : searchHits) {
                String index = hit.getIndex();
                String queryInfo = hit.getSourceAsString();
                SubjectDataVo info = JSON.parseObject(queryInfo, SubjectDataVo.class);
                info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
                info.setIndex(index);
                list.add(info);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }


    /**
     * 获取相似文章数
     *
     * @param subjectId     专题id
     * @param articleIdList 资讯id集合
     * @author lkg
     * @date 2024/4/10
     */
    public Map<String, Integer> getSimilarNumber(String subjectId, List<String> articleIdList) {
        Map<String, Integer> map = new HashMap<>();
        Map<String, String> markMap = getMark(subjectId, articleIdList);
        if (MapUtil.isNotEmpty(markMap)) {
            List<String> markList = new ArrayList<>(markMap.keySet());
            SearchRequest searchRequest = new SearchRequest(Constants.ES_REPEAT_OLD);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            //创建查询对象
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termsQuery("repeatMark", markList));
            searchSourceBuilder.size(0);
            searchSourceBuilder.trackTotalHits(true);
            TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_mark")
                    .field("repeatMark")
                    .order(BucketOrder.count(false))
                    .size(articleIdList.size());
            searchSourceBuilder.aggregation(aggregationBuilder);
            searchSourceBuilder.query(boolQuery);
            searchRequest.source(searchSourceBuilder);
            try {
                SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
                Aggregations aggregations = response.getAggregations();
                Terms groupSource = aggregations.get("group_mark");
                List<? extends Terms.Bucket> buckets = groupSource.getBuckets();
                if (CollectionUtils.isNotEmpty(buckets)) {
                    for (Terms.Bucket bucket : buckets) {
                        map.put(markMap.get(bucket.getKeyAsString()), (int) bucket.getDocCount());
                    }

                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return map;
    }

    /**
     * 获取资讯所在蔟(蔟中存储重复数据信息)
     *
     * @param subjectId     专题id
     * @param articleIdList 资讯id集合
     * @author lkg
     * @date 2024/4/10
     */
    private Map<String, String> getMark(String subjectId, List<String> articleIdList) {
        Map<String, String> map = new HashMap<>();
        if (CollectionUtils.isNotEmpty(articleIdList)) {
            List<String> idList = new ArrayList<>();
            for (String articleId : articleIdList) {
                idList.add(articleId.replace(subjectId, ""));
            }
            SearchRequest searchRequest = new SearchRequest(Constants.ES_REPEAT_OLD);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            //创建查询对象
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termsQuery("articleId", idList));
            boolQuery.must(QueryBuilders.termQuery("subjectId", subjectId));
            searchSourceBuilder.query(boolQuery);
            searchRequest.source(searchSourceBuilder);
            try {
                SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
                SearchHit[] hits = response.getHits().getHits();
                if (hits != null && hits.length != 0) {
                    for (SearchHit hit : hits) {
                        String queryInfo = hit.getSourceAsString();
                        RepeatHold info = JSONUtil.toBean(queryInfo, RepeatHold.class);
                        map.put(info.getRepeatMark(), info.getSubjectId() + info.getArticleId());
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return map;
    }


    /**
     * 按标题匹配数据并按来源分组
     *
     * @param title       标题
     * @param publishDate 发布时间
     * @author lkg
     * @date 2024/4/11
     */
    public List<String> groupByOrigin(String title, String publishDate) {
        List<String> originList = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.COLLECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //只返回分组聚合结果，不返回具体数据
        searchSourceBuilder.size(0);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
//        String[] arr = new String[]{"title"};
//        boolQuery.must(QueryBuilders.multiMatchQuery(title, arr));
        boolQuery.must(QueryBuilders.matchQuery("title", title));
        boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gt(EsDateUtil.esFieldDateFormat(publishDate)));
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_origin")
                .field("origin.keyword")
                .size(20)
                .order(BucketOrder.count(false));
        searchSourceBuilder.query(boolQuery);
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = searchResponse.getAggregations();
            Terms groupOrigin = aggregations.get("group_origin");
            List<? extends Terms.Bucket> buckets = groupOrigin.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                String origin = bucket.getKeyAsString();
                if (StringUtils.isNotEmpty(origin)) {
                    originList.add(origin);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return originList;
    }

    /**
     * 报告导出的资讯信息列表
     *
     * @param subjectIdList 专题id集合
     * @param searchWord    搜索词
     * @param position      搜索位置(title-标题;content-内容)
     * @param category      匹配度(1-模糊;2-精确)
     * @param articleIdList 资讯id集合
     * @param column        排序字段
     * @param order         排序方式(asc-正序;desc-倒序)
     * @param type          导出方式(1-摘要;2-正文)
     * @param pageSize      返回条数
     * @author lkg
     * @date 2024/4/11
     */
    public List<SubjectDataVo> exportDataList(List<String> subjectIdList, String searchWord, String position, Integer category,
                                              List<String> articleIdList, String column, String order, Integer type, Integer pageSize) {
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        if (CollectionUtils.isNotEmpty(articleIdList)) {
            pageSize = articleIdList.size();
        }
        //设置分页参数
        searchSourceBuilder.size(pageSize);
        if (column.equals("score")) {
            if (order.equals("asc")) {
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.ASC));
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
            } else if (order.equals("desc")) {
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.DESC));
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
            }
        } else if (column.equals("publishDate")) {
            if (order.equals("desc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.DESC));
            } else if (order.equals("asc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.ASC));
            }
        }
        String[] fetchFields = new String[]{"id", "title", "summary", "origin", "publishDate", "sourceAddress"};
        if (type == 2) {
            fetchFields = new String[]{"id", "title", "summary", "content", "origin", "publishDate", "sourceAddress"};
        }
        searchSourceBuilder.fetchSource(fetchFields, null);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("subjectId.keyword", subjectIdList));
        if (CollectionUtils.isNotEmpty(articleIdList)) {
            boolQuery.must(QueryBuilders.termsQuery("id", articleIdList));
        }
        if (StringUtils.isNotEmpty(searchWord)) {
            if (category == 1) {
                boolQuery.must(QueryBuilders.matchQuery(position, searchWord));
            } else if (category == 2) {
                boolQuery.must(QueryBuilders.matchPhraseQuery(position, searchWord));
            }
        }
        boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        List<SubjectDataVo> list = new ArrayList<>();
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            for (SearchHit hit : searchHits) {
                String queryInfo = hit.getSourceAsString();
                SubjectDataVo info = JSONUtil.toBean(queryInfo, SubjectDataVo.class);
                info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
                list.add(info);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }

    /**
     * 事件对应采集库的资讯分页列表
     *
     * @param eventDataCondition
     * @param sourceId
     * @param offset
     * @param pageSize
     * @param column
     * @param order
     * @param isAll
     * @param video
     * @param crawler
     * @param userId
     * @param tenantId
     * @param isSubject
     * @param source
     * @author lkg
     * @date 2024/5/6
     */
    public IPage<EventDataVO> collectPageList(EventDataCondition eventDataCondition, String sourceId, int offset, int pageSize, String column,
                                              String order, String isAll, String video, String crawler, String userId, String tenantId, String isSubject, String source) {
        SearchRequest searchRequest = new SearchRequest(Constants.COLLECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.from((offset - 1) * pageSize);
        //默认按照时间倒序排列
        if (column.equals("publishDate")) {
            if (order.equals("desc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
            } else if (order.equals("asc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
            }
        }
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        if ("1".equals(isAll)) {
            List<String> subjectIdList = new ArrayList<>();
            //判断是否为专题
            if (!"1".equals(isSubject)) {
                //不是专题
                if ("subjectType".equals(source)) {
                    //查询类别id的所有明细id
                    List<String> typeIds = new ArrayList<>();
                    if (StringUtils.isNotEmpty(eventDataCondition.getSubjectId()) && !"0".equals(eventDataCondition.getSubjectId())) {
                        typeIds = iSubjectTypeService.belowIdList(eventDataCondition.getSubjectId(), userId, null);
                    }
                    subjectIdList = eventService.selectSubjectByTypeIds(userId, typeIds);
                } else if ("customer".equals(source)) {
                    //查询该用户可以查看当前客户下面的专题
                    if (!StringUtils.isEmpty(eventDataCondition.getSubjectId()) && !"0".equals(eventDataCondition.getSubjectId()) && StringUtils.isEmpty(userId)) {
                        subjectIdList = eventService.selectSubjectWithCustomer(userId, null);
                    } else if (StringUtils.isNotBlank(userId)) {
                        subjectIdList = eventService.selectSubjectWithCustomer(userId, null);
                    } else {
                        subjectIdList = eventService.selectSubjectWithCustomer(null, tenantId);
                    }
                }
            } else {
                subjectIdList.add(eventDataCondition.getSubjectId());
            }
            //事件真正绑定的信息源id集合
            List<String> infoSourceIdList = new ArrayList<>();
            if (CollectionUtils.isNotEmpty(subjectIdList)) {
                infoSourceIdList = iSubjectInfoSourceMapService.subjectRealBindInfoSourceList(subjectIdList);
                //查询专题绑定的关键词组
                List<String> wordsIdList = eventService.bindKeyWordsIdList(subjectIdList);
                infoSourceIdList.addAll(wordsIdList);
            }
            //装配信息源的条件
            if (CollectionUtils.isNotEmpty(infoSourceIdList)) {
                BoolQueryBuilder childQuery = QueryBuilders.boolQuery();
                childQuery.should(QueryBuilders.termsQuery("sid", infoSourceIdList));
                boolQuery.must(childQuery);
            } else {
                return new Page<>();
            }
        } else {
            boolQuery.must(QueryBuilders.matchQuery("sid", sourceId));
        }
        if (StringUtils.isNotEmpty(eventDataCondition.getOrigin())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("origin", eventDataCondition.getOrigin()));
        }
        if (StringUtils.isNotEmpty(eventDataCondition.getSearch())) {
            boolQuery.must(QueryBuilders.multiMatchQuery(eventDataCondition.getSearch(), "title", "content"));
        }
        if (StringUtils.isNotEmpty(eventDataCondition.getContent())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("content", eventDataCondition.getContent()));
        }
        if (StringUtils.isNotEmpty(eventDataCondition.getTitle())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("title", eventDataCondition.getTitle()));
        }
        if (StringUtils.isNotEmpty(eventDataCondition.getSourceAddress())) {
            boolQuery.must(QueryBuilders.termQuery("sourceAddress", eventDataCondition.getSourceAddress()));
        }
        if (StringUtils.isNotEmpty(eventDataCondition.getId())) {
            boolQuery.must(QueryBuilders.termQuery("id", eventDataCondition.getId()));
        }
        if (StringUtils.isNotBlank(video)) {
            boolQuery.must(QueryBuilders.matchQuery("type", "video"));
        } else {
            boolQuery.mustNot(QueryBuilders.matchQuery("type", "video"));
        }
        if (StringUtils.isNotBlank(crawler)) {
            boolQuery.must(QueryBuilders.matchQuery("source", crawler));
        }
        //时间过滤筛选
        if (StringUtils.isNotBlank(eventDataCondition.getStartTime())) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gte(EsDateUtil.esFieldDateFormat(eventDataCondition.getStartTime())));
        }
        if (StringUtils.isNotBlank(eventDataCondition.getEndTime())) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(eventDataCondition.getEndTime())));
        } else {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(DateUtil.dateToString(new Date()))));
        }
        if (eventDataCondition.getLabelName() != null) {
            BoolQueryBuilder nestedBoolQueryBuilder = QueryBuilders.boolQuery();
            TermQueryBuilder relationIdQuery = QueryBuilders.termQuery("labels.relationName", eventDataCondition.getLabelName());
            nestedBoolQueryBuilder.should(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
            boolQuery.must(nestedBoolQueryBuilder);
        }
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        IPage<EventDataVO> pageData = new Page<>();
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            List<EventDataVO> list = new ArrayList<>();
            for (SearchHit hit : searchHits) {
                String queryInfo = hit.getSourceAsString();
                EventDataVO info = JSON.parseObject(queryInfo, EventDataVO.class);
                info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
                String hitIndex = hit.getIndex();
                info.setIndex(hitIndex);
                list.add(info);
            }
            pageData = new Page<>(offset, pageSize, searchResponse.getHits().getTotalHits().value);
            pageData.setRecords(list);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return pageData;
    }


    /**
     * 事件对应专题库的资讯分页列表
     *
     * @param userId
     * @param subjectInfo
     * @param video
     * @param offset
     * @param pageSize
     * @param column
     * @param order
     * @param crawler
     * @param isSubject
     * @param labelIds
     * @param socialCreditCodeList
     * @param sourceId
     * @param isCustomer
     * @author lkg
     * @date 2024/5/6
     */
    public IPage<DisplayInfo> subjectPageList(String userId, EventDataCondition subjectInfo, String video, int offset, int pageSize,
                                              String column, String order, String crawler, String isSubject,
                                              String labelIds, List<String> socialCreditCodeList, String sourceId, Integer isCustomer) throws Exception {
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.from((offset - 1) * pageSize);
        //默认按照置顶以及时间倒序排列
        //根据topNum正序查找，查询置顶数据
        searchSourceBuilder.sort("topNum", SortOrder.DESC);
        if (column.equals("score")) {
            if (order.equals("asc")) {
                searchSourceBuilder.sort("score", SortOrder.ASC);
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
            } else if (order.equals("desc")) {
                searchSourceBuilder.sort("score", SortOrder.DESC);
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
            }
        } else if (column.equals("publishDate")) {
            if (order.equals("desc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                searchSourceBuilder.sort("score", SortOrder.DESC);
            } else if (order.equals("asc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
                searchSourceBuilder.sort("score", SortOrder.ASC);
            }
        }
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();

        List<String> subjectIdList = new ArrayList<>();
        //判断是否是专题
        if ("1".equals(isSubject)) {
            if (StringUtils.isNotEmpty(subjectInfo.getSubjectId())) {
                subjectIdList.add(subjectInfo.getSubjectId());
            }
        } else {
            //该id其实是专题类别id
            //查询类别id的所有明细id
            List<String> typeIds = new ArrayList<>();
            String subjectTypeId = subjectInfo.getSubjectId();
            //获取当前登录用户
            UserVo currentUser = UserUtil.getLoginUser();
            if (isCustomer == 1) {
                subjectIdList = eventService.selectSubjectWithCustomer(null, null);
            } else {
                if (StringUtils.isNotEmpty(subjectTypeId) && !"0".equals(subjectTypeId)) {
                    typeIds = iSubjectTypeService.belowIdList(subjectTypeId, userId, null);
                }
                //判断是否是超级用户
                if (currentUser != null && currentUser.getCategory() == 1) {
                    subjectIdList = eventService.selectSubjectByTypeIds(null, typeIds);
                } else {
                    subjectIdList = eventService.selectSubjectByTypeIds(userId, typeIds);
                }
            }
        }
        if (CollectionUtils.isNotEmpty(subjectIdList)) {
            boolQuery.must(QueryBuilders.termsQuery("subjectId.keyword", subjectIdList));
        } else {
            return new Page<>();
        }
        if (StringUtils.isNotEmpty(subjectInfo.getSearch())) {
            boolQuery.must(QueryBuilders.multiMatchQuery(subjectInfo.getSearch(), "title", "content"));
        }
        if (StringUtils.isNotEmpty(subjectInfo.getContent())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("content", subjectInfo.getContent()));
        }
        if (StringUtils.isNotEmpty(subjectInfo.getTitle())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("title", subjectInfo.getTitle()));
        }
        if (StringUtils.isNotEmpty(subjectInfo.getOrigin())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("origin", subjectInfo.getOrigin()));
        }
        if (StringUtils.isNotEmpty(subjectInfo.getSourceAddress())) {
            boolQuery.must(QueryBuilders.termQuery("sourceAddress", subjectInfo.getSourceAddress()));
        }
        if (StringUtils.isNotEmpty(subjectInfo.getId())) {
            boolQuery.must(QueryBuilders.termQuery("id", subjectInfo.getId()));
        }
        if (subjectInfo.getCheckStatusList() != null && subjectInfo.getCheckStatusList().size() > 0) {
            List<Integer> list;
            list = subjectInfo.getCheckStatusList();
            if (!(list.size() == 1 && list.contains(4))) {
                boolQuery.must(QueryBuilders.termsQuery("checkStatus", subjectInfo.getCheckStatusList()));
            }
        }
        if (StringUtils.isNotBlank(video)) {
            boolQuery.must(QueryBuilders.matchQuery("type", "video"));
        } else {
            boolQuery.mustNot(QueryBuilders.matchQuery("type", "video"));
        }
        //只查询主条目
        boolQuery.mustNot(QueryBuilders.matchQuery("flag", "0"));
        //删除状态查询
        if (subjectInfo.getDeleteFlag() != 0) {
            boolQuery.must(QueryBuilders.matchQuery("deleteFlag", "1"));
        } else {
            boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        }
        //专题库类别筛选
        if (subjectInfo.getClassificationType() != null) {
            boolQuery.must(QueryBuilders.termQuery("classificationType", subjectInfo.getClassificationType()));
        }
        //正负面
        if (subjectInfo.getOrientation() != null) {
            boolQuery.must(QueryBuilders.termQuery("orientation", subjectInfo.getOrientation()));
        }
        //时间过滤筛选
        if (StringUtils.isNotBlank(subjectInfo.getStartTime())) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gte(EsDateUtil.esFieldDateFormat(subjectInfo.getStartTime())));
        }
        if (StringUtils.isNotBlank(subjectInfo.getEndTime())) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(subjectInfo.getEndTime())));
        } else {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(DateUtil.dateToString(new Date()))));
        }
        if (StringUtils.isNotBlank(crawler)) {
            boolQuery.must(QueryBuilders.matchQuery("source", crawler));
        }
        Set<String> relationIds = new HashSet<>();
        if (StringUtils.isNotEmpty(labelIds)) {
            relationIds.add(labelIds);
        }
        if (CollectionUtils.isNotEmpty(socialCreditCodeList)) {
            relationIds.addAll(socialCreditCodeList);
        }
        if (CollectionUtils.isNotEmpty(relationIds)) {
            BoolQueryBuilder nestedBoolQueryBuilder = QueryBuilders.boolQuery();
            for (String relationId : relationIds) {
                TermQueryBuilder relationIdQuery = QueryBuilders.termQuery("labels.relationId", relationId);
                nestedBoolQueryBuilder.should(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
            }
            boolQuery.must(nestedBoolQueryBuilder);
        }
        if (StringUtils.isNotEmpty(subjectInfo.getArea())) {
            String[] regionArr = subjectInfo.getArea().split(",");
            for (String regionId : regionArr) {
                BoolQueryBuilder nestedRegionBoolQueryBuilder = QueryBuilders.boolQuery();
                TermQueryBuilder relationIdQuery = QueryBuilders.termQuery("labels.relationId", regionId);
                nestedRegionBoolQueryBuilder.should(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
                boolQuery.must(nestedRegionBoolQueryBuilder);
            }
        }
        if (subjectInfo.getLabelName() != null) {
            BoolQueryBuilder nestedBoolQueryBuilder = QueryBuilders.boolQuery();
            TermQueryBuilder relationIdQuery = QueryBuilders.termQuery("labels.relationName", subjectInfo.getLabelName());
            nestedBoolQueryBuilder.should(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
            boolQuery.must(nestedBoolQueryBuilder);
        }
        if (StringUtils.isNotBlank(sourceId)) {
            boolQuery.must(QueryBuilders.termQuery("sid", sourceId));
        }
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
        SearchHit[] searchHits = searchResponse.getHits().getHits();
        List<DisplayInfo> list = new ArrayList<>();
        List<LabelModelVo> labelModelVos = eventService.selectLabelModelBySubjectId(subjectIdList);
        Map<String, List<LabelModelVo>> modelMap = labelModelVos.stream().collect(Collectors.groupingBy(LabelModelVo::getSubjectId));
        for (SearchHit hit : searchHits) {
            String queryInfo = hit.getSourceAsString();
            DisplayInfo info = JSONUtil.toBean(queryInfo, DisplayInfo.class);
            info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
            QueryWrapper<CollectionMap> query = Wrappers.query();
            query.eq("user_id", userId);
            query.eq("article_id", info.getId());
            int count = collectionMapService.count(query);
            if (count > 0) {
                info.setYnCollect(true);
            }
            //标签处理
            List<LabelModelVo> modelVoList = modelMap.get(info.getSubjectId());
            formatLabel(modelVoList, info);
            String index = hit.getIndex();
            info.setIndex(index);
            list.add(info);
        }
        IPage<DisplayInfo> pageData = new Page<>(offset, pageSize, searchResponse.getHits().getTotalHits().value);

        pageData.setRecords(list);
        return pageData;
    }

    /**
     * 获取信息源时间段内的采集量
     *
     * @param sid       信息源id
     * @param startDate 开始日期
     * @param endDate   结束日期
     * @param type      分类(1-不聚合;2-聚合)
     * @author lkg
     * @date 2024/4/25
     */
    public NumVO count(String sid, String startDate, String endDate, Integer type) {
        NumVO countVO = new NumVO();
        SearchRequest searchRequest = new SearchRequest(Constants.COLLECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("sid", sid));
        if (StringUtils.isNotBlank(startDate) || StringUtils.isNotBlank(endDate)) {
            RangeQueryBuilder rangeQueryBuilder = QueryBuilders.rangeQuery("createDate");
            if (StringUtils.isNotBlank(startDate)) {
                rangeQueryBuilder.from(EsDateUtil.esFieldDateFormat(startDate));
            }
            if (StringUtils.isNotBlank(endDate)) {
                rangeQueryBuilder.to(EsDateUtil.esFieldDateFormat(endDate));
            }
            boolQuery.filter(rangeQueryBuilder);
        }
        searchSourceBuilder.query(boolQuery);
        if (type == 2) {
            DateHistogramAggregationBuilder aggregation = AggregationBuilders.dateHistogram("group_date")
                    .field("createDate")
                    .calendarInterval(DateHistogramInterval.DAY)
                    .format("yyyy-MM-dd");
            searchSourceBuilder.aggregation(aggregation);
        }
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            long num = searchResponse.getHits().getTotalHits().value;
            countVO.setCount(num);
            if (type == 2) {
                List<NumVO> list = new ArrayList<>();
                Aggregations aggregations = searchResponse.getAggregations();
                ParsedDateHistogram groupHour = aggregations.get("group_date");
                List<? extends Histogram.Bucket> buckets = groupHour.getBuckets();
                if (CollectionUtils.isNotEmpty(buckets)) {
                    for (Histogram.Bucket bucket : buckets) {
                        NumVO vo = new NumVO();
                        vo.setKey(bucket.getKeyAsString());
                        vo.setCount(bucket.getDocCount());
                        list.add(vo);
                    }
                }
                countVO.setChildren(list);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return countVO;
    }

    public Map<String, Map<String, Long>> regetFromEs(List<String> sidList, String startDate, String endDate) {
        SearchRequest searchRequest = new SearchRequest(Constants.COLLECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.size(0);
        //时间范围筛选
        RangeQueryBuilder rangeQueryBuilder = QueryBuilders
                .rangeQuery("createDate")
                .from(startDate)
                .to(endDate);
        searchSourceBuilder.query(rangeQueryBuilder);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("sid", sidList));
        boolQuery.must(rangeQueryBuilder);
        //按时间，sid分组
        //es 查询天的时间格式
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_sid")
                .size(10000)
                .field("sid")
                .subAggregation(AggregationBuilders.dateHistogram("group_date")
                        .field("createDate")
                        .calendarInterval(DateHistogramInterval.DAY)
                        .format("yyyy-MM-dd"));
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        Map<String, Map<String, Long>> map = new HashMap<>();
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            //获取分组桶
            Aggregations aggregations = searchResponse.getAggregations();
            //获取id分组集合
            Terms parsedStringTerms = aggregations.get("group_sid");

            List<? extends Terms.Bucket> buckets = parsedStringTerms.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                //key的数据，过滤sid为空的
                String key = bucket.getKey().toString();
                if (key == null || "null".equals(key) || "".equals(key)) {
                    continue;
                }
                //获取时间分组集合
                Aggregations bucketAggregations = bucket.getAggregations();
                ParsedDateHistogram groupdateAgr = bucketAggregations.get("group_date");
                List<? extends Histogram.Bucket> dateBucket = groupdateAgr.getBuckets();
                Map<String, Long> temMap = new HashMap<>();
                for (Histogram.Bucket date : dateBucket) {
                    String dateStr = date.getKeyAsString();
                    long num = date.getDocCount();
                    temMap.put(dateStr, num);
                }
                map.put(key, temMap);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return map;
    }

    public List<SubjectAnalysis> eventAnalysisByOrigin(String subjectId) {
        List<SubjectAnalysis> dataList = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("subjectId.keyword", subjectId));
        boolQuery.must(QueryBuilders.termsQuery("origin.keyword", Arrays.asList(EVENT_ANALYSIS_ORIGIN)));
        //未删除
        boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse;
        try {
            searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] hits = searchResponse.getHits().getHits();
            for (SearchHit hit : hits) {
                Map<String, Object> sourceAsMap = hit.getSourceAsMap();
                SubjectAnalysis subjectAnalysis = new SubjectAnalysis();
                Object id = sourceAsMap.get("id");
                if (ObjectUtils.isNotEmpty(id)) {
                    subjectAnalysis.setDataId(id.toString());
                }
                Object title = sourceAsMap.get("title");
                if (ObjectUtils.isNotEmpty(title)) {
                    subjectAnalysis.setTitle(title.toString());
                }
                Object origin = sourceAsMap.get("origin");
                if (ObjectUtils.isNotEmpty(origin)) {
                    subjectAnalysis.setOrigin(origin.toString());
                }
                Object publishDate = sourceAsMap.get("publishDate");
                if (ObjectUtils.isNotEmpty(origin)) {
                    String dateFormat = EsDateUtil.esFieldDateMapping(publishDate.toString());
                    subjectAnalysis.setPublishDate(DateUtil.stringToDate(dateFormat, "yyyy-MM-dd HH:mm:ss"));
                }
                subjectAnalysis.setSubjectId(subjectId);
                dataList.add(subjectAnalysis);
            }
        } catch (Exception e) {
            log.info("获取专题-{}-下资讯，查询es库失败", subjectId);
            e.printStackTrace();
        }
        return dataList;
    }

    public List<EventDigDataVO> pageListOfColumn(String columnId, Integer checkStatus, String startTime, String endTime, Integer pageSize) {
        List<EventDigDataVO> dataList = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        searchSourceBuilder.size(pageSize);
        //排序方式 按入专题库时间升序
        searchSourceBuilder.sort("processDate", SortOrder.ASC);
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        List<String> columnIds = Arrays.asList(columnId.split(","));
        BoolQueryBuilder columnQuery = getQueryByChanelIds(columnIds);
        boolQuery.must(columnQuery);
        if (StringUtils.isNotEmpty(startTime)) {
            boolQuery.filter(QueryBuilders.rangeQuery("processDate").gt(EsDateUtil.esFieldDateFormat(startTime)));
        }
        if (StringUtils.isNotEmpty(endTime)) {
            boolQuery.filter(QueryBuilders.rangeQuery("processDate").lte(EsDateUtil.esFieldDateFormat(endTime)));
        }
        if (checkStatus != null) {
            if (checkStatus == 4) {
                //删除状态查询
                boolQuery.must(QueryBuilders.termQuery("deleteFlag", 1));
            } else {
                boolQuery.must(QueryBuilders.termQuery("checkStatus", checkStatus));
            }
        }
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] hits = searchResponse.getHits().getHits();
            if (hits != null && hits.length > 0) {
                for (SearchHit hit : hits) {
                    String index = hit.getIndex();
                    String sourceAsString = hit.getSourceAsString();
                    EventDigDataVO eventDigDataVO = JSON.parseObject(sourceAsString, EventDigDataVO.class);
                    eventDigDataVO.setPublishDate(EsDateUtil.esFieldDateMapping(eventDigDataVO.getPublishDate()));
                    eventDigDataVO.setProcessDate(EsDateUtil.esFieldDateMapping(eventDigDataVO.getProcessDate()));
                    eventDigDataVO.setIndex(index);
                    dataList.add(eventDigDataVO);
                }
            }
        } catch (Exception e) {
            log.info("获取栏目-{}-下资讯，查询es库失败", columnId);
            e.printStackTrace();
        }
        return dataList;
    }


    /**
     * 媒体观点/专家观点信息列表
     *
     * @param eventId    事件id
     * @param relationId 关联标签id
     * @author lkg
     * @date 2024/10/10
     */
    public List<EventViewVO> viewpointList(String eventId, String relationId) {
        List<EventViewVO> dataList = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //排序方式 按发布时间升序
        searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        searchSourceBuilder.trackTotalHits(true);
        String[] fetchFields = new String[]{"id", "title", "origin", "publishDate", "subjectId", "sourceAddress","labels"};
        searchSourceBuilder.fetchSource(fetchFields, null);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("subjectId.keyword", eventId));
        boolQuery.must(QueryBuilders.nestedQuery("labels", QueryBuilders.termQuery("labels.relationId", relationId), ScoreMode.None));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] hits = searchResponse.getHits().getHits();
            if (hits != null && hits.length > 0) {
                for (SearchHit hit : hits) {
                    String index = hit.getIndex();
                    String sourceAsString = hit.getSourceAsString();
                    EventViewVO eventViewVO = JSON.parseObject(sourceAsString, EventViewVO.class);
                    eventViewVO.setPublishDate(EsDateUtil.esFieldDateMapping(eventViewVO.getPublishDate()));
                    eventViewVO.setIndex(index);
                    List<Label> labels = eventViewVO.getLabels();
                    String professionName = null;
                    for (Label label : labels) {
                        String labelRelationId = label.getRelationId();
                        if (StringUtils.isEmpty(labelRelationId)) {
                            professionName = label.getRelationName();
                            break;
                        }
                    }
                    eventViewVO.setProfessionName(professionName);
                    dataList.add(eventViewVO);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return dataList;
    }

    public List<SubjectDataVo> listByIds(List<String> ids) {
        List<SubjectDataVo> dataList = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("id", ids));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] hits = searchResponse.getHits().getHits();
            if (hits != null && hits.length > 0) {
                for (SearchHit hit : hits) {
                    String sourceAsString = hit.getSourceAsString();
                    SubjectDataVo subjectDataVo = JSONObject.parseObject(sourceAsString, SubjectDataVo.class);
                    dataList.add(subjectDataVo);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return dataList;
    }


    /**
     * 栏目查询
     *
     * @param channelIds 栏目id集合
     * @return
     */
    private BoolQueryBuilder getQueryByChanelIds(List<String> channelIds) {
        LambdaQueryWrapper<SysBaseChannel> baseChannelLambdaQueryWrapper = Wrappers.lambdaQuery();
        baseChannelLambdaQueryWrapper.in(SysBaseChannel::getId, channelIds);
        List<SysBaseChannel> channelList = channelService.list(baseChannelLambdaQueryWrapper);
        //验证频道和标签是否绑定专题或者标签
        boolean validation = validation(channelList);
        if (!validation) {
            return null;
        }
        BoolQueryBuilder channelAllBoolQueryBuilder = QueryBuilders.boolQuery();
        for (SysBaseChannel channel : channelList) {
            //获取频道关联的标签
            LambdaQueryWrapper<ClbChannelLabelMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(ClbChannelLabelMap::getChannelId, channel.getId());
            List<ClbChannelLabelMap> clbChannelLabelMapList = channelLabelMapService.list(queryWrapper);
            Set<String> labelTypeSet = new HashSet<>();
            Set<String> labelIdSet = new HashSet<>();
            if (CollectionUtils.isNotEmpty(clbChannelLabelMapList)) {
                for (ClbChannelLabelMap clbChannelLabelMap : clbChannelLabelMapList) {
                    if (1 == clbChannelLabelMap.getType()) {
                        labelTypeSet.add(clbChannelLabelMap.getLabelTypeMark());
                    } else if (2 == clbChannelLabelMap.getType()) {
                        labelIdSet.add(clbChannelLabelMap.getLabelId());
                    }
                }
            }
            String subjectIds = channel.getSubjectIds();
            String relation = channel.getRelation();
            BoolQueryBuilder channelOneBoolQueryBuilder = QueryBuilders.boolQuery();
            BoolQueryBuilder subjectIdQueryBuilder = QueryBuilders.boolQuery();
            BoolQueryBuilder labelQueryBuilder = QueryBuilders.boolQuery();
            if (StringUtils.isNotEmpty(subjectIds)) {
                String[] subjectIdArr = subjectIds.split(",");
                subjectIdQueryBuilder.filter(QueryBuilders.termsQuery("subjectId.keyword", subjectIdArr));
            }
            if (CollectionUtils.isNotEmpty(labelIdSet)) {
                TermsQueryBuilder termsQueryBuilder = QueryBuilders.termsQuery("labels.relationId", labelIdSet);
                NestedQueryBuilder relationIdNestedQueryBuilder = QueryBuilders.nestedQuery("labels", termsQueryBuilder, ScoreMode.None);
                labelQueryBuilder.must(relationIdNestedQueryBuilder);
            }
            if (CollectionUtils.isNotEmpty(labelTypeSet)) {
                BoolQueryBuilder markQuery = QueryBuilders.boolQuery();
                for (String labelMark : labelTypeSet) {
                    markQuery.should(QueryBuilders.matchQuery("labels.labelMark", labelMark));
                }
                NestedQueryBuilder markNestedQueryBuilder = QueryBuilders.nestedQuery("labels", markQuery, ScoreMode.None);
                labelQueryBuilder.must(markNestedQueryBuilder);
            }
            if (StringUtils.isNotEmpty(relation)) {
                if ("and".equals(relation)) {
                    channelOneBoolQueryBuilder.must(subjectIdQueryBuilder).must(labelQueryBuilder);
                } else if ("or".equals(relation)) {
                    channelOneBoolQueryBuilder.should(subjectIdQueryBuilder).should(labelQueryBuilder);
                }
            } else {
                if (StringUtils.isNotEmpty(subjectIds)) {
                    channelOneBoolQueryBuilder.must(subjectIdQueryBuilder);
                }
                if (CollectionUtils.isNotEmpty(labelTypeSet) || CollectionUtils.isNotEmpty(labelIdSet)) {
                    channelOneBoolQueryBuilder.must(labelQueryBuilder);
                }
            }
            channelAllBoolQueryBuilder.should(channelOneBoolQueryBuilder);
        }
        return channelAllBoolQueryBuilder;
    }


    private boolean validation(List<SysBaseChannel> channelList) {
        boolean flag = false;
        List<String> channelIds = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(channelList)) {
            for (SysBaseChannel sysBaseChannel : channelList) {
                channelIds.add(sysBaseChannel.getId());
                if (StringUtils.isNotBlank(sysBaseChannel.getSubjectIds())) {
                    flag = true;
                    break;
                }
            }
            if (!flag) {
                //获取频道关联的标签
                LambdaQueryWrapper<ClbChannelLabelMap> queryWrapper = Wrappers.lambdaQuery();
                queryWrapper.in(ClbChannelLabelMap::getChannelId, channelIds);
                List<ClbChannelLabelMap> clbChannelLabelMapList = channelLabelMapService.list(queryWrapper);
                if (clbChannelLabelMapList.size() > 0) {
                    flag = true;
                }
            }
        }
        return flag;
    }


    private void formatLabel(List<LabelModelVo> labelModelVos, DisplayInfo info) {
        if (CollectionUtils.isNotEmpty(labelModelVos)) {
            List<Label> list = info.getLabels();
            List<LabelInfo> labelInfos = new ArrayList<>();
            //获取专题打的标签
            for (LabelModelVo labelModelVo : labelModelVos) {
                LabelInfo labelInfo = new LabelInfo();
                labelInfo.setLabelId(labelModelVo.getLabelId());
                labelInfo.setLabelName(labelModelVo.getLabelName());
                labelInfo.setLabelType(labelModelVo.getLabelType());
                labelInfo.setLabelMark(labelModelVo.getLabelMark());
                List<Label> labelList = new ArrayList<>();
                if (list != null && list.size() > 0) {
                    for (Label label : list) {
                        if (StringUtils.isNotBlank(label.getLabelMark())) {
                            if (label.getLabelMark().contains(labelModelVo.getLabelMark()) || ("company_label".equals(labelModelVo.getLabelType()) && "company_label".equals(label.getLabelRemarks()))) {
                                labelList.add(label);
                            }
                        }
                    }
                }
                labelInfo.setLabelList(labelList);
                labelInfos.add(labelInfo);
            }
            info.setLabelInfos(labelInfos);
        }
    }


    //格式化数据
    private List<SubjectDataVo> formatData(SearchHits searchHits) {
        List<SubjectDataVo> list = new ArrayList<>();
        if (searchHits != null) {
            SearchHit[] hits = searchHits.getHits();
            for (SearchHit hit : hits) {
                String index = hit.getIndex();
                String sourceAsString = hit.getSourceAsString();
                SubjectDataVo subjectDataVo = JSON.parseObject(sourceAsString, SubjectDataVo.class);
                subjectDataVo.setPublishDate(EsDateUtil.esFieldDateMapping(subjectDataVo.getPublishDate()));
                subjectDataVo.setIndex(index);
                list.add(subjectDataVo);
            }
        }
        return list;
    }

    //获取es查询结果
    private SearchHits getEsResult(SearchSourceBuilder searchSourceBuilder, BoolQueryBuilder boolQuery, String subjectId) {
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //未删除
        boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse;
        try {
            searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            return searchResponse.getHits();
        } catch (Exception e) {
            log.info("获取专题-{}-下资讯，查询es库失败", subjectId);
            e.printStackTrace();
        }
        return null;
    }
}
