package com.zzsn.event.controller.yjzx;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.clb.common.model.task.dto.titr.KeyWordsDTO;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.Event;
import com.zzsn.event.entity.KeyWords;
import com.zzsn.event.entity.SysDictItem;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.*;
import com.zzsn.event.service.impl.ConfigurationMessageService;
import com.zzsn.event.util.*;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.*;
import com.zzsn.event.vo.es.Label;
import com.zzsn.event.xxljob.service.IXxlJobInfoService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * 事件
 *
 * @Author: jeecg-boot
 * @Date: 2024-03-14
 * @Version: V1.0
 */
@Slf4j
@Api(tags = "事件")
@RestController
@RequestMapping("/data")
public class EventDataController {
    @Autowired
    private IEventService eventService;
    @Autowired
    private EsService esService;
    @Autowired
    private EventSimpleService eventSimpleService;
    @Autowired
    private IXxlJobInfoService iXxlJobInfoService;
    @Autowired
    private IKeyWordsService keyWordsService;
    @Autowired
    private InformationService informationService;
    @Autowired
    private ConfigurationMessageService configurationMessageService;
    @Autowired
    private RedisUtil redisUtil;
    @Resource
    private KafkaTemplate<String, String> kafkaTemplate;
    @Autowired
    private EsOpUtil esOpUtil;
    @Autowired
    private ObsUtil obsUtil;
    @Autowired
    private SysDictItemService dictItemService;
    @Autowired
    private AnalysisService analysisService;

    @Value("${translate.url:}")
    private String TRANSLATE_URL;
    @Value("${kafka.topic.event.run:}")
    private String EVENT_MODEL_KAFKA_CHANNEL;

    @Value("${clb.eventAdd:}")
    private String CLB_EVENT_ADD;

    @Value("${clb.eventEdit:}")
    private String CLB_EVENT_EDIT;

    @Value("${clb.eventDelete:}")
    private String CLB_EVENT_DELETE;

    @Value("${clb.clearData:}")
    private String CLB_CLEAR_DATA;


    private static final String prefix = "http://obs.ciglobal.cn/";

    /**
     * 上传文件(事件报告跳转智能写作是调用)
     *
     * @param file 文件
     * @author lkg
     * @date 2024/8/16
     */
    @PostMapping("/uploadFile")
    public Result<?> uploadFile(MultipartFile file) {
        String downloadUrl = null;
        try {
            if (file != null) {
                String filename = file.getOriginalFilename();
                if (StringUtils.isNotEmpty(filename)) {
                    String[] split = filename.split("\\.");
                    String objectKey = "event/report/" + split[0] + "_" + System.currentTimeMillis() + "." + split[1];
                    obsUtil.uploadFile(objectKey, file.getInputStream());
                    downloadUrl = prefix + objectKey;
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return Result.OK(downloadUrl);
    }

    /**
     * 删除文件(研究中心智能写作保存文件后,调用接口删除平台的报告文件)
     *
     * @param param 参数
     * @author lkg
     * @date 2024/8/16
     */
    @PostMapping("/deleteFile")
    public Result<?> deleteFile(@RequestBody JSONObject param) {
        if (param != null) {
            String path = param.get("path").toString();
            String objectKey = path.replace(prefix, "");
            obsUtil.delFile(objectKey);
        }
        return Result.OK();
    }

    /**
     * 事件分页列表-门户
     *
     * @param category   栏目类型(1-全部;2-自动追踪;3-我创建的;4-我关注的)
     * @param eventName  事件名称
     * @param eventType  事件分类id
     * @param labelField 搜索字段(标签)
     * @param labelName  搜索词(标签)
     * @param order      排序字段
     * @param orderType  排序方式
     * @param pageNo     偏移量
     * @param pageSize   返回条数
     * @author lkg
     * @date 2024/4/8
     */
    @ApiOperation(value = "事件-分页列表查询", notes = "事件-分页列表查询")
    @GetMapping(value = "/pageList")
    public Result<?> pageList(@RequestParam(defaultValue = "1476527644425682945") String projectId,
                              @RequestParam(name = "category") Integer category,
                              @RequestParam(name = "eventId", required = false) String eventId,
                              @RequestParam(name = "eventName", required = false) String eventName,
                              @RequestParam(name = "eventType", required = false) String eventType,
                              @RequestParam(name = "labelField", required = false) String labelField,
                              @RequestParam(name = "labelName", required = false) String labelName,
                              @RequestParam(name = "order", required = false) String order,
                              @RequestParam(name = "orderType", required = false) String orderType,
                              @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                              @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        IPage<EventFrontVO> pageList = new Page<>();
        if (category == 1) {
            pageList = eventService.frontAllPageList(projectId, eventId, eventName, eventType, labelField, labelName, order, orderType, pageNo, pageSize);
        } else if (category == 3) {
            pageList = eventService.frontOwnerPageList(projectId, eventName, eventType, labelField, labelName, order, orderType, pageNo, pageSize);
        } else if (category == 4) {
            pageList = eventService.frontCollectPageList(projectId, eventName, eventType, labelField, labelName, order, orderType, pageNo, pageSize);
        }
        return Result.OK(pageList);
    }

    /**
     * 新增事件(页面简化版)
     *
     * @param addEventVO 事件信息
     * @author lkg
     * @date 2024/7/22
     */
    //@PostMapping("/simpleSaveEvent")
    public Result<?> addEvent(@RequestBody AddEventVO addEventVO) {
        if (StringUtils.isNotEmpty(CLB_EVENT_ADD)) {
            UserVo currentUser = UserUtil.getLoginUser();
            KeywordsVO keywordsVO = addEventVO.getKeywordsVO();
            addEventVO.setTenant(currentUser.getTenant());
            if (keywordsVO != null) {
                // 调用克虏宝事件新增接口
                JSONObject params = JSONObject.from(addEventVO);
                params.put("username", currentUser.getUsername());
                try {
                    String resultStr = HttpUtil.doPost(CLB_EVENT_ADD, params, 1000 * 30);
                    JSONObject jsonObject = JSONObject.parseObject(resultStr);
                    Integer code = jsonObject.getInteger("code");
                    if (code != null && code.equals(Constants.SUCCESS_CODE)) {
                        JSONObject resultMap = jsonObject.getJSONObject("result");
                        String id = resultMap.getString("id");
                        String eventCode = resultMap.getString("eventCode");
                        JSONObject clbKeyWords = resultMap.getJSONObject("keyWords");

                        // 事件id和code使用clb中生成的
                        addEventVO.setId(id);
                        addEventVO.setEventCode(eventCode);
                        Event event = eventSimpleService.simpleSave(addEventVO);
                        // 关键词使用clb中返回
                        KeyWords keyWords = new KeyWords();
                        keyWords.setId(clbKeyWords.getString("id"));
                        keyWords.setWordsCode(clbKeyWords.getString("wordsCode"));
                        keyWords.setWordsName(clbKeyWords.getString("wordsName"));
                        keyWords.setKeyWord(keywordsVO.getKeyword());
                        keyWords.setExclusionWord(keywordsVO.getExclusionWord());
                        keyWords.setStatus("1");
                        keyWords.setCreateTime(new Date());
                        keyWordsService.saveKeyword(event.getId(), keyWords);
                        return Result.OK();
                    } else {
                        log.error("调用克虏宝新增返回失败");
                        return Result.FAIL(500, "保存事件信息失败!");
                    }
                } catch (Exception e) {
                    log.error("调用克虏宝新增事件异常：{}", e);
                    return Result.FAIL(500, "保存事件信息失败!");
                }
            } else {
                return Result.FAIL(500, "关键词不能为空");
            }
        } else {
            KeywordsVO keywordsVO = addEventVO.getKeywordsVO();
            if (keywordsVO != null) {
                Event event = eventSimpleService.simpleSave(addEventVO);
                KeyWords keyWords = keyWordsService.saveKeyword(event, keywordsVO.getKeyword(), keywordsVO.getExclusionWord());
                CompletableFuture.runAsync(() -> {
                    iXxlJobInfoService.eventInsert(event);
                    //关键词
                    KeyWordsDTO redisKeywordDTO = new KeyWordsDTO();
                    BeanUtils.copyProperties(keyWords, redisKeywordDTO);
                    redisKeywordDTO.setStartTime(event.getStartTime());
                    redisKeywordDTO.setEndTime(event.getEndTime());
                    redisKeywordDTO.setSearchEngines(new ArrayList<>(Constants.DEFAULT_SEARCH_ENGINE.values()));
                    redisUtil.set(Constants.KEY_WORDS_TO_REDIS_PREFIX + keyWords.getWordsCode(), redisKeywordDTO);
                    //插入xxljob
                    iXxlJobInfoService.keyWordsInsert(redisKeywordDTO);
                    //为了立即响应，关键词新增时放入一个首次录入消息队列
                    kafkaTemplate.send(Constants.KEY_WORDS_COLLECT_TOPIC, JSON.toJSONString(redisKeywordDTO));
                    //通知采集
                    configurationMessageService.bindKeyWordsEventSend(event.getId());
                    kafkaTemplate.send(EVENT_MODEL_KAFKA_CHANNEL, event.getEventCode());
                });
                return Result.OK();
            } else {
                return Result.FAIL(500, "关键词不能为空");
            }
        }
    }

    @PostMapping("/simpleSaveEvent")
    public Result<?> addEvent_new(@RequestBody AddEventVO addEventVO) {
        KeywordsVO keywordsVO = addEventVO.getKeywordsVO();
        if (keywordsVO != null) {
            UserVo currentUser = UserUtil.getLoginUser();
            AddEventParam eventParam = new AddEventParam();
            BeanUtils.copyProperties(addEventVO, eventParam);
            eventParam.setCreateBy(currentUser.getUsername());
            eventParam.setOpen(true);
            eventParam.setTenant(currentUser.getTenant());
            Event event = eventService.saveMain(eventParam);
            //新增关键词
            keyWordsService.saveBaseAndMap(event, keywordsVO.getKeyword(), keywordsVO.getExclusionWord());
            CompletableFuture.runAsync(() -> {
                try {
                    //插入xxlJob
                    iXxlJobInfoService.eventInsert(event);
                    //通知采集
                    configurationMessageService.bindKeyWordsEventSend(event.getId());
                    //事件分析
                    kafkaTemplate.send(EVENT_MODEL_KAFKA_CHANNEL, event.getEventCode());
                } catch (Exception e) {
                    e.printStackTrace();
                }
                //立马执行一次事件分析
                analysisService.regenerate(event.getId(),null);
            });
            return Result.OK();
        } else {
            return Result.FAIL(500, "关键词不能为空");
        }
    }


    /**
     * 编辑事件(页面简化版)
     *
     * @param addEventVO 事件信息
     * @author lkg
     * @date 2024/7/22
     */
    //@PostMapping("/simpleUpdateEvent")
    public Result<?> updateEvent(@RequestBody AddEventVO addEventVO) {
        if (StringUtils.isNotEmpty(CLB_EVENT_EDIT)) {
            UserVo currentUser = UserUtil.getLoginUser();
            KeywordsVO keywordsVO = addEventVO.getKeywordsVO();
            addEventVO.setTenant(currentUser.getTenant());
            if (keywordsVO != null) {
                // 调用克虏宝编辑接口
                JSONObject params = JSONObject.from(addEventVO);
                params.put("username", currentUser.getUsername());
                try {
                    String resultStr = HttpUtil.doPost(CLB_EVENT_EDIT, params, 1000 * 30);
                    JSONObject jsonObject = JSONObject.parseObject(resultStr);
                    Integer code = jsonObject.getInteger("code");
                    if (code != null && code.equals(Constants.SUCCESS_CODE)) {
                        // 本地编辑接口
                        eventSimpleService.updateMain(addEventVO);
                        keyWordsService.update(Wrappers.<KeyWords>lambdaUpdate().eq(KeyWords::getId, keywordsVO.getId())
                                .set(KeyWords::getKeyWord, keywordsVO.getKeyword())
                                .set(KeyWords::getExclusionWord, keywordsVO.getExclusionWord()));
                    } else {
                        log.error("调用克虏宝编辑返回失败");
                        return Result.FAIL(500, "编辑事件信息失败!");
                    }
                } catch (Exception e) {
                    log.error("调用克虏宝编辑事件异常：{}", e);
                    return Result.FAIL(500, "编辑事件信息失败!");
                }
                return Result.OK();
            } else {
                return Result.FAIL(500, "关键词不能为空");
            }
        } else {
            KeywordsVO keywordsVO = addEventVO.getKeywordsVO();
            if (keywordsVO != null) {
                eventSimpleService.updateMain(addEventVO);
                keyWordsService.update(Wrappers.<KeyWords>lambdaUpdate().eq(KeyWords::getId, keywordsVO.getId())
                        .set(KeyWords::getKeyWord, keywordsVO.getKeyword())
                        .set(KeyWords::getExclusionWord, keywordsVO.getExclusionWord()));
                CompletableFuture.runAsync(() -> {
                    //关键词
                    KeyWordsDTO redisKeywordDTO = new KeyWordsDTO();
                    BeanUtils.copyProperties(keywordsVO, redisKeywordDTO);
                    redisKeywordDTO.setKeyWord(keywordsVO.getKeyword());
                    redisKeywordDTO.setStartTime(addEventVO.getStartTime());
                    redisKeywordDTO.setEndTime(addEventVO.getEndTime());
                    redisKeywordDTO.setSearchEngines(new ArrayList<>(Constants.DEFAULT_SEARCH_ENGINE.values()));
                    redisUtil.set(Constants.KEY_WORDS_TO_REDIS_PREFIX + keywordsVO.getWordsCode(), redisKeywordDTO);
                    //为了立即响应，关键词新增时放入一个首次录入消息队列
                    kafkaTemplate.send(Constants.KEY_WORDS_COLLECT_TOPIC, JSON.toJSONString(redisKeywordDTO));
                    //通知采集
                    configurationMessageService.bindKeyWordsEventSend(addEventVO.getId());
                });
                return Result.OK();
            } else {
                return Result.FAIL(500, "关键词不能为空");
            }
        }
    }


    @PostMapping("/simpleUpdateEvent")
    public Result<?> updateEvent_new(@RequestBody AddEventVO addEventVO) {
        KeywordsVO keywordsVO = addEventVO.getKeywordsVO();
        AddEventParam addEventParam = new AddEventParam();
        BeanUtils.copyProperties(addEventVO, addEventParam);
        addEventParam.setTenant(UserUtil.getLoginUser().getTenant());
        if (keywordsVO != null) {
            UserVo currentUser = UserUtil.getLoginUser();
            addEventParam.setUpdateTime(new Date());
            addEventParam.setUpdateBy(currentUser.getUsername());
            eventService.updateMain(addEventParam);
            keyWordsService.updateBaseAndMap(addEventParam, keywordsVO.getKeyword(), keywordsVO.getExclusionWord());
            CompletableFuture.runAsync(() -> {
                //通知采集
                configurationMessageService.bindKeyWordsEventSend(addEventParam.getId());
            });
            return Result.OK();
        } else {
            return Result.FAIL(500, "关键词不能为空");
        }
    }

    /**
     * 事件配置校验
     *
     * @param addEventParam 参数
     * @author lkg
     * @date 2025/2/5
     */
    @PostMapping("/configVerification")
    public Result<?> configVerification(@RequestBody AddEventParam addEventParam) {
        Boolean verification = eventService.configVerification(addEventParam);
        return Result.OK(verification);
    }

    /**
     * 清空专题数据
     *
     * @param eventId 专题id
     * @author lkg
     * @date 2025/2/6
     */
    //@GetMapping("/clearData")
    public Result<?> clearData(@RequestParam String eventId) {
        if (StringUtils.isNotEmpty(CLB_CLEAR_DATA)) {
            try {
                Map<String, String> params = new HashMap<>();
                params.put("eventId", eventId);
                UserVo loginUser = UserUtil.getLoginUser();
                Map<String, String> headers = new HashMap<>();
                params.put("loginUser", JSON.toJSONString(loginUser));
                HttpUtil.doGet(CLB_CLEAR_DATA, params, headers, "utf-8");
            } catch (Exception e) {
                return Result.FAIL(500, "调用克虏宝清空专题数据异常!");
            }
        }
        eventService.clearSubjectData(eventId);
        return Result.OK("正在清空数据");
    }

    @GetMapping("/clearData")
    public Result<?> clearData_new(@RequestParam String eventId) {
        eventService.clearSubjectData(eventId);
        return Result.OK("正在清空数据");
    }

    /**
     * 通过id删除
     *
     * @param param:id 事件id
     * @return
     */
    //@PostMapping(value = "/simpleDeleteEvent")
    public Result<?> delete(@RequestBody JSONObject param) {
        if (StringUtils.isNotEmpty(CLB_EVENT_DELETE)) {
            try {
                String id = param.getString("id");
                if (StringUtils.isBlank(id)) {
                    return Result.FAIL(500, "删除事件信息失败，入参不完整!");
                }
                // 调用克虏宝删除接口
                Map<String, String> params = new HashMap<>();
                params.put("id", id);
                HttpUtil.doGet(CLB_EVENT_DELETE, params, null, "utf-8");
                // 调用本地删除
                EventVO eventVO = eventService.queryInfo(id);
                eventService.deleteMain(id);
                CompletableFuture.runAsync(() -> {
                    KeywordsVO keywordsVO = eventVO.getKeywordsVO();
                    if (keywordsVO != null) {
                        String wordsCode = keywordsVO.getWordsCode();
                        keyWordsService.remove(Wrappers.<KeyWords>lambdaQuery().eq(KeyWords::getWordsCode, wordsCode));
                    }
                });
            } catch (Exception e) {
                log.error("调用克虏宝删除事件异常：{}", e);
                return Result.FAIL(500, "删除事件信息失败!");
            }
            return Result.OK();
        } else {
            String id = param.getString("id");
            EventVO eventVO = eventService.queryInfo(id);
            eventService.deleteMain(id);
            CompletableFuture.runAsync(() -> {
                iXxlJobInfoService.deleteByInfosourceCode(eventVO.getEventCode());
                KeywordsVO keywordsVO = eventVO.getKeywordsVO();
                if (keywordsVO != null) {
                    String wordsCode = keywordsVO.getWordsCode();
                    keyWordsService.remove(Wrappers.<KeyWords>lambdaQuery().eq(KeyWords::getWordsCode, wordsCode));
                    redisUtil.del(Constants.KEY_WORDS_TO_REDIS_PREFIX + wordsCode);
                    iXxlJobInfoService.deleteByInfosourceCode(wordsCode);
                }
            });
            return Result.OK();
        }
    }

    @PostMapping(value = "/simpleDeleteEvent")
    public Result<?> delete_new(@RequestBody JSONObject param) {
        String id = param.getString("id");
        eventService.deleteMain(id);
        CompletableFuture.runAsync(() -> {
            EventVO eventVO = eventService.queryInfo(id);
            iXxlJobInfoService.deleteByInfosourceCode(eventVO.getEventCode());
            //删除关键词
            keyWordsService.removeBaseAndMap(id);
        });
        return Result.OK();
    }


    @GetMapping("/labelList")
    public Result<?> labelList() {
        List<SysDictItem> dictItemList = dictItemService.listByDictCode("eventView");
        List<SysDictItem> collect = dictItemList.stream().filter(e -> !e.getItemText().equals("无观点")).collect(Collectors.toList());
        return Result.OK(collect);
    }


    /**
     * 事件资讯分页列表
     *
     * @param projectId  项目id
     * @param subjectId  事件id
     * @param searchWord 搜索词
     * @param position   搜索位置(title-标题;content-内容)
     * @param category   匹配度(1-模糊;2-精确)
     * @param origin     来源
     * @param labelId    标签id
     * @param startTime  开始时间
     * @param endTime    结束时间
     * @param column     排序字段
     * @param order      排序方式(asc-正序;desc-倒序)
     * @param pageNo     当前页
     * @param pageSize   返回条数
     * @author lkg
     * @date 2024/4/10
     */
    @ApiOperation(value = "专题信息列表-分页列表查询", notes = "专题信息列表-分页列表查询")
    @GetMapping(value = "/listArticle")
    public Result<?> queryPageList(@RequestParam(name = "projectId", defaultValue = "1476527644425682945") String projectId,
                                   @RequestParam(name = "eventId", required = false) String subjectId,
                                   @RequestParam(name = "searchWord", required = false) String searchWord,
                                   @RequestParam(name = "position", required = false) String position,
                                   @RequestParam(name = "category", required = false) Integer category,
                                   @RequestParam(name = "origin", required = false) String origin,
                                   @RequestParam(name = "labelId", required = false) String labelId,
                                   @RequestParam(name = "startTime", required = false) String startTime,
                                   @RequestParam(name = "endTime", required = false) String endTime,
                                   @RequestParam(name = "column", defaultValue = "publishDate") String column,
                                   @RequestParam(name = "order", defaultValue = "desc") String order,
                                   @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                   @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        UserVo currentUser = UserUtil.getLoginUser();
        String username = currentUser.getUsername();
        List<String> subjectIdList = new ArrayList<>();
        if (StringUtils.isNotEmpty(subjectId)) {
            subjectIdList.add(subjectId);
        } else {
            List<EventExcelVO> frontList = eventService.frontAllList(projectId, username, null, null, null, null, null, null);
            frontList.forEach(e -> subjectIdList.add(e.getId()));
        }
        //获取数据
        IPage<SubjectDataVo> pageList = esService.frontListByPage(subjectIdList, searchWord, position, category, origin, labelId, startTime, endTime, column, order, pageNo, pageSize);
        return Result.OK(pageList);
    }


    /**
     * 导出excel
     *
     * @param projectId  项目id
     * @param ids        资讯id集合
     * @param subjectId  事件id
     * @param searchWord 搜索词
     * @param position   搜索位置(title-标题;content-内容)
     * @param category   匹配度(1-模糊;2-精确)
     * @param labelId    标签id
     * @param startTime  开始时间
     * @param endTime    结束时间
     * @param column     排序字段
     * @param order      排序方式(asc-正序;desc-倒序)
     * @author lkg
     * @date 2024/4/10
     */
    @ApiOperation(value = "专题信息列表-导出excel", notes = "专题信息列表-导出excel")
    @GetMapping(value = "/exportExcel")
    public void exportExcel(@RequestParam(name = "projectId", defaultValue = "1476527644425682945") String projectId,
                            @RequestParam(name = "ids", required = false) List<String> ids,
                            @RequestParam(name = "eventId") String subjectId,
                            @RequestParam(name = "searchWord", required = false) String searchWord,
                            @RequestParam(name = "position", required = false) String position,
                            @RequestParam(name = "category", required = false) Integer category,
                            @RequestParam(name = "labelId", required = false) String labelId,
                            @RequestParam(name = "startTime", required = false) String startTime,
                            @RequestParam(name = "endTime", required = false) String endTime,
                            @RequestParam(name = "column", defaultValue = "publishDate") String column,
                            @RequestParam(name = "order", defaultValue = "desc") String order,
                            HttpServletResponse response) {
        UserVo currentUser = UserUtil.getLoginUser();
        String username = currentUser.getUsername();
        List<String> subjectIdList = new ArrayList<>();
        if (StringUtils.isNotEmpty(subjectId)) {
            subjectIdList.add(subjectId);
        } else {
            List<EventExcelVO> frontList = eventService.frontAllList(projectId, username, null, null, null, null, null, null);
            frontList.forEach(e -> subjectIdList.add(e.getId()));
        }
        //栏目字典
        List<SysDictItem> dictItemList = dictItemService.listByDictCode("eventView");
        try {
            long timestamp = System.currentTimeMillis();
            String filename = subjectId + "_" + timestamp + ".xlsx";
            String[] arr = new String[]{"标题", "摘要", "正文", "来源", "发布时间", "原文链接", "所属栏目"};
            List<String> headers = Arrays.asList(arr);
            SXSSFWorkbook workbook = new SXSSFWorkbook();
            int pageSize = 1000;
            for (int i = 1; ; i++) {
                List<SubjectDataVo> informationList = esService.exportList(subjectIdList, ids, searchWord, position, category, labelId, startTime, endTime, column, order, i, pageSize);
                log.info("本次循环-{}，数据量为-{}", i, informationList.size());
                if (CollectionUtils.isEmpty(informationList)) {
                    break;
                }
                List<List<String>> rows = formatExportData(informationList, dictItemList);
                BigExcelExportUtil.exportExcelData(workbook, i - 1, headers, rows, "sheet" + i);
                log.info("第【{}】个sheet页写入成功", i);
            }
            // 将Workbook写入字节流
            ByteArrayOutputStream outStream = new ByteArrayOutputStream();
            workbook.write(outStream);
            // 将字节流转换为InputStream
            byte[] bytes = outStream.toByteArray();
            filename = URLEncoder.encode(filename, "UTF-8").replace("+", " ");
            response.setHeader("Access-Control-Expose-Headers", "Content-Disposition");
            response.setHeader("Content-Disposition", "attachment;filename=" + filename);
            response.setContentLength(bytes.length);
            OutputStream out = response.getOutputStream();
            out.write(bytes);
            out.flush();
            out.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    /**
     * 资讯详情
     *
     * @return
     */
    @GetMapping(value = "/articleDetail")
    public Result<?> articleDetail(@RequestParam(required = false) String index, @RequestParam String id) {
        SubjectDataVo subjectDataVo = esService.queryInfo(index, id);
        return Result.OK(subjectDataVo);
    }

    /**
     * 批量删除资讯
     *
     * @param list 信息集合
     * @author lkg
     * @date 2024/9/14
     */
    @PostMapping("/batchDelete")
    public Result<?> batchDelete(@RequestBody List<JSONObject> list) {
        Map<String, String> updateFieldMap = new HashMap<>();
        updateFieldMap.put("deleteFlag", "1");
        List<CompletableFuture<Void>> asyncList = new ArrayList<>();
        for (JSONObject jsonObject : list) {
            asyncList.add(CompletableFuture.runAsync(() -> {
                String id = jsonObject.getString("id");
                String index = jsonObject.getString("index");
                esOpUtil.updateById(index, id, updateFieldMap);
            }));
        }
        //等待全部执行完
        CompletableFuture.allOf(asyncList.toArray(new CompletableFuture[0])).join();
        return Result.OK();
    }

    /**
     * 翻译
     *
     * @param transferData 但翻译文本
     * @author lkg
     * @date 2024/8/21
     */
    @PostMapping("/translate")
    public Result<?> translate(@RequestBody SubjectDataVo transferData) {
        TranslateVO translateVO = new TranslateVO();
        translateVO.setFrom("auto");
        translateVO.setTo("zh-CN");
        List<String> content = new ArrayList<>();
        content.add(transferData.getTitle());
        content.add(transferData.getSummary());
        content.add(transferData.getContent());
        translateVO.setContent(content);
        List<String> data = new ArrayList<>();
        try {
            String s = HttpUtil.doPost(TRANSLATE_URL, ObjectUtil.objectToJSONObject(translateVO), 10000);
            data = JSONArray.parseArray(s, String.class);
        } catch (Exception e) {
            log.error("翻译请求失败！");
            e.printStackTrace();
        }
        return Result.OK(data);
    }

    /**
     * 单篇文章热词
     *
     * @param index  索引名称
     * @param id     资讯id
     * @param number 热词数量
     * @author lkg
     * @date 2024/4/10
     */
    @GetMapping(value = "/hotWords")
    public Result<?> articleList(@RequestParam(value = "index", required = false) String index,
                                 @RequestParam("id") String id,
                                 @RequestParam(name = "number", defaultValue = "200") Integer number) {
        if (StringUtils.isEmpty(index)) {
            index = Constants.SUBJECT_INDEX;
        }
        List<StatisticsKeyWordVo> words = informationService.hotWords(index, id, number);
        return Result.OK(words);
    }

    /**
     * 相关推荐
     *
     * @param subjectId 专题id
     * @param id        资讯id
     * @param title     标题
     * @param pageSize  返回条数
     * @author lkg
     * @date 2024/4/10
     */
    @GetMapping(value = "/recommendList")
    public Result<?> recommendList(@RequestParam(name = "subjectId") String subjectId,
                                   @RequestParam(name = "id") String id,
                                   @RequestParam(name = "title") String title,
                                   @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                   @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        List<SubjectDataVo> recommendList = informationService.recommendList(subjectId, id, title, pageNo, pageSize);
        return Result.OK(recommendList);
    }

    private List<List<String>> formatExportData(List<SubjectDataVo> list, List<SysDictItem> dictItemList) {
        List<List<String>> rows = new ArrayList<>();
        Map<String, String> eventViewMap = dictItemList.stream().collect(Collectors.toMap(SysDictItem::getItemValue, SysDictItem::getItemText));
        for (SubjectDataVo subjectDataVo : list) {
            List<String> row = new ArrayList<>();
            row.add(StringUtils.isNotEmpty(subjectDataVo.getTitle()) ? subjectDataVo.getTitle() : "");
            row.add(StringUtils.isNotEmpty(subjectDataVo.getSummary()) ? subjectDataVo.getSummary() : "");
            row.add(StringUtils.isNotEmpty(subjectDataVo.getContent()) ? subjectDataVo.getContent() : "");
            row.add(StringUtils.isNotEmpty(subjectDataVo.getOrigin()) ? subjectDataVo.getOrigin() : "");
            row.add(StringUtils.isNotEmpty(subjectDataVo.getPublishDate()) ? subjectDataVo.getPublishDate() : "");
            row.add(StringUtils.isNotEmpty(subjectDataVo.getSourceAddress()) ? subjectDataVo.getSourceAddress() : "");
            List<Label> labels = subjectDataVo.getLabels();
            if (CollectionUtils.isNotEmpty(labels)) {
                List<Label> collect = labels.stream().filter(e -> eventViewMap.containsKey(e.getRelationId())).collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(collect)) {
                    row.add(collect.stream().map(e -> eventViewMap.get(e.getRelationId())).collect(Collectors.joining(",")));
                } else {
                    row.add("");
                }
            } else {
                row.add("");
            }
            rows.add(row);
        }
        return rows;
    }
}
