package com.zzsn.event.controller;

import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.LabelEntity;
import com.zzsn.event.entity.SubjectAnalysis;
import com.zzsn.event.service.*;
import com.zzsn.event.util.CalculateUtil;
import com.zzsn.event.util.HttpUtil;
import com.zzsn.event.vo.CountVO;
import com.zzsn.event.vo.PropagationPathVo;
import com.zzsn.event.vo.SubjectDataVo;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

/**
 * 事件分析
 *
 * @author lkg
 * @date 2024/1/24
 */
@RestController
@RequestMapping("/analysis")
public class EventAnalysisController {

    @Autowired
    private EsStatisticsService esStatisticsService;
    @Autowired
    private LabelEntityService labelEntityService;
    @Autowired
    private AnalysisService analysisService;
    @Autowired
    private SubjectAnalysisService subjectAnalysisService;
    @Autowired
    private EsService esService;

    @Value(("${serviceProject.url:}"))
    private String SERVICE_PROJECT_URL;


    /**
     * 3.4 总体分析
     *
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/1/24
     */
    @GetMapping("/total")
    public Result<?> totalAndMax(@RequestParam String subjectId, @RequestParam String startTime, @RequestParam String endTime) {
        Map<String, Object> map = esStatisticsService.totalAndMax(subjectId, null, null);
        long hours = DateUtil.between(DateUtil.parseDateTime(startTime), DateUtil.parseDateTime(endTime), DateUnit.HOUR);
        map.put("duration", hours);
        Object count = map.get("totalCount");
        String divide = CalculateUtil.divide(String.valueOf(count), String.valueOf(hours), 2);
        map.put("spread", divide);
        String mainReport = esStatisticsService.mainReport(subjectId);
        map.put("mainReport", mainReport);
        return Result.OK(map);
    }

    /**
     * 热门信息列表
     *
     * @param subjectId 专题id
     * @param size      返回条数
     * @author lkg
     * @date 2024/4/10
     */
    @GetMapping("/hotList")
    public Result<?> hotList(@RequestParam(name = "subjectId") String subjectId,
                             @RequestParam(name = "size", defaultValue = "10") Integer size) {
        String[] fetchFields = new String[]{"id", "title", "origin", "publishDate", "sourceAddress"};
        List<SubjectDataVo> pageList = esService.pageList(subjectId, null, null, fetchFields, 2, 1, size);
        if (CollectionUtils.isNotEmpty(pageList)) {
            List<String> idList = new ArrayList<>();
            pageList.forEach(e -> idList.add(e.getId()));
            Map<String, Integer> similarNumber = esService.getSimilarNumber(subjectId, idList);
            for (SubjectDataVo subjectDataVo : pageList) {
                String dataId = subjectDataVo.getId();
                Integer count = similarNumber.get(dataId);
                if (count == null) {
                    count = 0;
                }
                subjectDataVo.setSimilarNumber(count);
            }
        }
        return Result.OK(pageList);
    }

    /**
     * 3.1 传播路径
     *
     * @param eventId 事件id
     */
    @GetMapping("/propagationPath")
    public Result<?> propagationPath(@RequestParam String eventId) {
        PropagationPathVo pathVo = analysisService.propagationPath(eventId);
        return Result.OK(pathVo);
    }

    /**
     * 3.2 事件脉络
     *
     * @param subjectId 专题id
     * @param fakeNum   专题事件脉络展示 伪事件脉络 的资讯数量阈值
     * @return com.zzsn.subjectAnalysis.common.Result
     */
    @GetMapping("/eventContext")
    public Result<?> eventContext(@RequestParam String subjectId,
                                  @RequestParam(value = "fakeNum", required = false) Integer fakeNum) {
        if (fakeNum == null) {
            fakeNum = Constants.FAKE_NUM;
        }
        List<SubjectAnalysis> cacheList = analysisService.eventContext(subjectId, fakeNum);
        return Result.OK(cacheList);
    }

    /**
     * 3.8 观点
     *
     * @author lkg
     * @date 2024/1/12
     */
    @ApiOperation(value = "专题信息列表-分页列表查询", notes = "专题信息列表-分页列表查询")
    @GetMapping(value = "/listEventAnalysis")
    public Result<?> listEventAnalysis(@RequestParam(name = "eventId") String eventId,
                                       @RequestParam(name = "type", defaultValue = "0") Integer type) {
        LambdaQueryWrapper<SubjectAnalysis> wrapper = new LambdaQueryWrapper<SubjectAnalysis>()
                .eq(SubjectAnalysis::getSubjectId, eventId)
                .eq(SubjectAnalysis::getCategory, 1);
        if (4 == type) {
            wrapper.eq(SubjectAnalysis::getType, 4);
        } else {
            wrapper.ne(SubjectAnalysis::getType, 4);
        }
        List<SubjectAnalysis> list = subjectAnalysisService.list(wrapper);
        return Result.OK(list);
    }

    /**
     * 3.3 热词查询
     *
     * @return
     */
    @ApiOperation(value = "热词查询", notes = "热词查询")
    @GetMapping(value = "/keywordsCount")
    public Object keywordsCount(@RequestParam String subjectId, @RequestParam Integer pageSize) {
        try {
            JSONObject params = new JSONObject();
            params.put("subjectId", subjectId);
            params.put("pageSize", pageSize);
            String url = SERVICE_PROJECT_URL + "event/keywordsCount";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }


    /**
     * 3.5 情感判断分析
     *
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param type      1-按小时;2-按天
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/orientation")
    public Result<?> orientation(@RequestParam String subjectId, @RequestParam String startTime,
                                 @RequestParam String endTime, @RequestParam Integer type) {
        List<CountVO> list = new ArrayList<>();
        String labelTypeId = "1631119596744265729";
        List<LabelEntity> labelEntities = labelEntityService.listByType(labelTypeId);
        AtomicLong total = new AtomicLong();
        labelEntities.forEach(e -> {
            CompletableFuture<CountVO> async = CompletableFuture.supplyAsync(() -> {
                CountVO countVO = esStatisticsService.orientation(subjectId, e.getId(), startTime, endTime, type);
                total.addAndGet(countVO.getValue());
                supply(countVO, startTime, endTime, type);
                return countVO;
            });
            try {
                CountVO countVO = async.get();
                list.add(countVO);
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        });
        for (CountVO countVO : list) {
            long value = countVO.getValue();
            long totalCount = total.get();
            String divide = CalculateUtil.divide(String.valueOf(value), String.valueOf(totalCount));
            String percentage = "0%";
            if (StringUtils.isNotEmpty(divide)) {
                percentage = CalculateUtil.percentage(Double.parseDouble(divide), false);
            }
            countVO.setPercentage(percentage);
        }
        return Result.OK(list);
    }

    /**
     * 3.6 信息传播走势
     *
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param type      1-按小时;2-按天
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/flowData")
    public Result<?> flowData(@RequestParam String subjectId, @RequestParam String startTime,
                              @RequestParam String endTime, @RequestParam Integer type) {
        AtomicLong total = new AtomicLong();
        CountVO countVO = esStatisticsService.flowData(subjectId, startTime, endTime);
        total.addAndGet(countVO.getValue());
        supply(countVO, startTime, endTime, type);
        long value = countVO.getValue();
        long totalCount = total.get();
        String divide = CalculateUtil.divide(String.valueOf(value), String.valueOf(totalCount));
        String percentage = "0%";
        if (StringUtils.isNotEmpty(divide)) {
            percentage = CalculateUtil.percentage(Double.parseDouble(divide), false);
        }
        countVO.setPercentage(percentage);
        return Result.OK(countVO);
    }


    /**
     * 按来源分析
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/origin")
    public Result<?> origin(@RequestParam String subjectId) {
        List<CountVO> list = esStatisticsService.origin(subjectId, null, null);
        return Result.OK(list);
    }


    //补充缺失的时间
    private void supply(CountVO countVO, String startTime, String endTime, Integer type) {
        List<CountVO> list = new ArrayList<>();
        List<CountVO> children = countVO.getChildren();
        Map<String, CountVO> map = children.stream().collect(Collectors.toMap(CountVO::getName, item -> item, (k1, k2) -> k2));
        DateTime startDate = DateUtil.parseDateTime(startTime);
        DateTime endDate = DateUtil.parseDateTime(endTime);
        List<DateTime> rangeToList = new ArrayList<>();
        String format = null;
        if (type == 1) {
            rangeToList = DateUtil.rangeToList(startDate, endDate, DateField.HOUR_OF_DAY);
            format = "yyyy-MM-dd HH";
        } else if (type == 2) {
            rangeToList = DateUtil.rangeToList(startDate, endDate, DateField.DAY_OF_YEAR);
            format = "yyyy-MM-dd";
        }
        for (DateTime dateTime : rangeToList) {
            String date = DateUtil.format(dateTime, format);
            if (map.containsKey(date)) {
                list.add(countVO);
            } else {
                CountVO vo = new CountVO();
                vo.setName(date);
                vo.setValue(0L);
                list.add(vo);
            }
        }
        countVO.setChildren(list);
    }
}
