package com.zzsn.event.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.Event;
import com.zzsn.event.enums.CodePrefixEnum;
import com.zzsn.event.mapper.EventMapper;
import com.zzsn.event.producer.ProduceInfo;
import com.zzsn.event.service.*;
import com.zzsn.event.util.*;
import com.zzsn.event.vo.AddEventParam;
import com.zzsn.event.vo.KeyWordsDTO;
import com.zzsn.event.vo.KeyWordsPage;
import com.zzsn.event.vo.SubjectKafkaVo;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDate;
import java.util.*;

/**
 * @Description: 事件
 * @Author: jeecg-boot
 * @Date: 2024-03-14
 * @Version: V1.0
 */
@Service
@Slf4j
public class EventServiceImpl extends ServiceImpl<EventMapper, Event> implements IEventService {
    @Value("${hotWords.extractUrl}")
    private String extractHotWordsUrl;
    @Autowired
    private CodeGenerateUtil codeGenerateUtil;
    @Autowired
    private IKeyWordsService iKeyWordsService;
    @Autowired
    private ISubjectSearchEnginesMapService iSubjectSearchEnginesMapService;
    @Autowired
    private ISubjectInfoSourceMapService iSubjectInfoSourceMapService;
    @Autowired
    private ISubjectKeywordsMapService iSubjectKeywordsMapService;
    @Autowired
    private ProduceInfo produceInfo;
    @Autowired
    private RedisUtil redisUtil;
    @Value("${files.storage}")
    String filesStorage;

    @Value("${scoreRule.weekScore}")
    Integer weekScore;
    @Value("${scoreRule.monthScore}")
    Integer monthScore;
    @Value("${scoreRule.yearScore}")
    Integer yearScore;
    @Value("${scoreRule.beforeYearScore}")
    Integer beforeYearScore;

    @Override
    public void compute() {
        //get the events need to compute
        List<Event> list = this.list(new LambdaQueryWrapper<Event>().eq(Event::getStatus, 1));
        for (Event event : list) {
            //total
            Integer totalScore = 0;
            Integer totalHot = getHot(Constants.MAX_TOTAL_HOT_SCORE, totalScore);
            //media
            Integer mediaScore = 0;
            Integer mediaHot = getHot(Constants.MAX_MEDIA_HOT_SCORE, mediaScore);
            //wechat
            Integer wechatScore = 0;
            Integer wechatHot = getHot(Constants.MAX_WECHAT_HOT_SCORE, wechatScore);
            //others
            Integer otherScore = 0;
            Integer otherHot = getHot(Constants.MAX_OTHER_HOT_SCORE, otherScore);
            event.setTotalHot(totalHot);
            event.setMediaHot(mediaHot);
            event.setWechatHot(wechatHot);
            event.setOtherHot(otherHot);
            this.updateById(event);

        }

    }


    private Integer getHot(String key, Integer Score) {
        Object total = redisUtil.get(Constants.MAX_TOTAL_HOT_SCORE);
        if (null != total) {
            Integer maxTotalScore = (Integer) total;
            if (Score > maxTotalScore) {
                redisUtil.set(Constants.MAX_TOTAL_HOT_SCORE, Score);
                return 100;
            } else {
                return Score / maxTotalScore * 100;
            }
        } else {
            redisUtil.set(Constants.MAX_TOTAL_HOT_SCORE, Score);
            return 100;
        }
    }

    /**
     * 计算热度
     *
     * @param weekTimes       周内次数
     * @param monthTimes      月内次数
     * @param yearTimes       年内次数
     * @param beforeYearTimes 年外次数
     */
    private Integer computeScore(Integer weekTimes, Integer monthTimes, Integer yearTimes, Integer beforeYearTimes) {
        // latest week data
        // latest month data
        // latest year data
        // data before a year
        return weekTimes * weekScore + monthTimes * monthScore + yearTimes * yearScore + beforeYearTimes * beforeYearScore;
    }

    @Override
    public List<SubjectKafkaVo> progressList(Date disableDate) {
        return this.baseMapper.prosessList(disableDate);
    }

    @Override
    public List<SubjectKafkaVo> eventSubjectList() {
        return this.baseMapper.eventSubjectList(2);
    }

    @Override
    public IPage<Event> pageList(Event event, Integer pageNo, Integer pageSize) {
        Integer offset = (pageNo - 1) * pageSize;
        if (pageNo == 0) {
            offset = 0;
        }
        List<Event> pageList = baseMapper.pageList(event, offset, pageSize);
        //获取总条数
        Integer count = baseMapper.totalCount(event);
        IPage<Event> pageData = new Page<>(pageNo, pageSize, count);
        pageData.setRecords(pageList);
        return pageData;
    }

    @Override
    public void extractHotWords(AddEventParam event) {
        if (!StringUtils.isEmpty(event.getExtractHotWords()) && "1".equals(event.getExtractHotWords())) {
            Map<String, String> param = new HashMap<>();
            param.put("status", "0,1,2,3");
            param.put("timeFiled", "ALL");
            param.put("subjectId", event.getId());
            HttpUtil.doGet(extractHotWordsUrl, param, "utf-8");
        }
    }

    @Override
    public Event saveMain(AddEventParam addEventParam) {
        String cron = null;
        Event event = new Event();
        //事件专题的默认分析规则参数-必填
        BeanUtils.copyProperties(addEventParam, event);
        event.setIncreAnaRule(20);
        event.setTotalAnaRule(50);
        event.setTimeAnaRule(5);
        String subjectCode = codeGenerateUtil.geneCodeNo(CodePrefixEnum.SUBJECT_DEFAULT.getValue());
        event.setEventCode(subjectCode);
        cron = CronUtil.getRandomCron();
        event.setCron(cron);
        baseMapper.insert(event);
        return event;
    }

    @Override
    public void updateMain(AddEventParam addEventParam) {
        Event subject = new Event();
        BeanUtils.copyProperties(addEventParam, subject);
        //先记录老的信息状态
        Event oldSubject = baseMapper.selectById(subject.getId());
        baseMapper.updateById(subject);

        //判断开始时间和结束时间是否发生变动
        if ((StringUtils.isEmpty(addEventParam.getStartTime()) && StringUtils.isEmpty(oldSubject.getStartTime())) ||
                (!StringUtils.isEmpty(addEventParam.getStartTime()) && !StringUtils.isEmpty(oldSubject.getStartTime())
                        && oldSubject.getStartTime().compareTo(addEventParam.getStartTime()) == 0)) {

        } else {
            //查询出该专题绑定的关键词组
            List<KeyWordsPage> keyWordsPages = bindKeyWordsList(subject.getId());
            List<String> keyWordIds = new ArrayList<>();
            for (KeyWordsPage keyWordsPage : keyWordsPages) {
                keyWordIds.add(keyWordsPage.getId());
            }
            //更新redis中关键词时间
            updateRedisKeyWordsDate(addEventParam.getId(), keyWordIds);
        }
    }

    @Override
    public List<KeyWordsPage> bindKeyWordsList(String subjectId) {
        return iKeyWordsService.bindKeyWordsList(subjectId);
    }

    @Override
    public void deleteMain(String id) {
        //删除与信息源的关联关系
        iSubjectInfoSourceMapService.delete(id);
        //删除与关键词组的关联关系
        iSubjectKeywordsMapService.delete(id);
        baseMapper.deleteById(id);
    }

    @Override
    public String upload(MultipartFile file) {

        try {
            String fileName = file.getOriginalFilename();  // 获取文件名称
            long fileSize = file.getSize();  // get file size
            log.info("文件上传：" + fileName);
            String filePath = getFilePath() + fileName;
            //byte[] bytes = file.getBytes();  // get file in bytes
            //Files.write(Paths.get(filePath), bytes);
            String path = ClassUtils.getDefaultClassLoader().getResource("").getPath();
            file.transferTo(new File(filesStorage + filePath));
            log.info("文件上传成功：" + fileName + "---" + filePath);
            return filePath;
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.getMessage());
        }
        // if the operation of file uploading is unsuccessful, an error occurs
        log.info("上传文件时出现问题！");
        return null;
    }

    //生成文件夹路径
    private String getFilePath() {
        LocalDate currentDate = LocalDate.now();
        String current = currentDate.toString().replace("-", "");
        String filePath = current + "/";

        //判断文件夹是否存在，不存在创建
        Path directory = Paths.get(filesStorage + filePath);
        if (!Files.exists(directory)) {
            try {
                Files.createDirectories(directory);
                log.info("文件夹创建成功：" + filePath);
            } catch (IOException e) {
                log.error("文件夹创建失败：" + filePath);
                e.printStackTrace();
            }
        }
        return filePath;
    }

    //更新专题绑定的关键词组的缓存信息
    private void updateRedisKeyWordsDate(String subjectId, List<String> keyWordsIds) {
        for (String keyWordsId : keyWordsIds) {
            KeyWordsDTO keyWordsDTO = iSubjectKeywordsMapService.selectMinByKeyWordsId(keyWordsId);
            KeyWordsDTO redisKeyWordsDTO = (KeyWordsDTO) redisUtil.get(Constants.KEY_WORDS_TO_REDIS_PREFIX + keyWordsDTO.getWordsCode());
            int count = iSubjectKeywordsMapService.selectCountByKeyWordsId(keyWordsId);
            KeyWordsDTO keyWordsDTO1;
            if (count <= 0) {
                keyWordsDTO1 = iSubjectKeywordsMapService.selectMaxByKeyWordsId(keyWordsId);
                redisKeyWordsDTO.setStartTime(keyWordsDTO.getStartTime());
                redisKeyWordsDTO.setEndTime(keyWordsDTO1.getEndTime());
            } else {
                if (redisKeyWordsDTO == null) {
                    redisKeyWordsDTO = keyWordsDTO;
                } else {
                    redisKeyWordsDTO.setEndTime(null);
                    redisKeyWordsDTO.setStartTime(keyWordsDTO.getStartTime());
                }
            }
            //查询出该专题绑定了哪些搜索引擎
            List<String> stringList = iSubjectSearchEnginesMapService.querySearchList(subjectId);
            redisKeyWordsDTO.setSearchEngines(stringList);
            redisUtil.set(Constants.KEY_WORDS_TO_REDIS_PREFIX + redisKeyWordsDTO.getWordsCode(), redisKeyWordsDTO);
            //立即执行一次
            produceInfo.sendKeyWordsInfoSourceMsg(redisKeyWordsDTO);
        }
    }

}
