package com.zzsn.event.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.Event;
import com.zzsn.event.entity.SubjectAnalysis;
import com.zzsn.event.producer.ProduceInfo;
import com.zzsn.event.service.AnalysisService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.util.ObjectUtil;
import com.zzsn.event.util.RedisUtil;
import com.zzsn.event.util.RestUtil;
import com.zzsn.event.vo.*;
import com.zzsn.event.xxljob.service.IXxlJobInfoService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;
import org.springframework.web.util.WebUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.concurrent.CompletableFuture;

/**
 * 事件
 * @Author: jeecg-boot
 * @Date:   2024-03-14
 * @Version: V1.0
 */
@Slf4j
@Api(tags="事件")
@RestController
@RequestMapping("/event/event")
public class EventController  {
	@Autowired
	private IEventService eventService;
	 @Value(("${serviceProject.url:}"))
	 private String SERVICE_PROJECT_URL;

	 @Autowired
	 private RedisUtil redisUtil;
	 @Autowired
	 private AnalysisService analysisService;
	@Autowired
	private IXxlJobInfoService iXxlJobInfoService;
	@Autowired
	private ProduceInfo produceInfo;
	
	/**
	 * 分页列表查询
	 *
	 * @param event
	 * @param pageNo
	 * @param pageSize
	 * @param req
	 * @return
	 */
	@ApiOperation(value="事件-分页列表查询", notes="事件-分页列表查询")
	@GetMapping(value = "/list")
	public Result<?> queryPageList(Event event,
								   @RequestParam(name="pageNo", defaultValue="1") Integer pageNo,
								   @RequestParam(name="pageSize", defaultValue="10") Integer pageSize,
								   HttpServletRequest req) {

		IPage<Event> pageList = eventService.pageList(event, pageNo,pageSize);
		return Result.OK(pageList);
	}
	
	/**
	 * 添加
	 *
	 * @param eventParam
	 * @return
	 */
	@ApiOperation(value="事件-添加", notes="事件-添加")
	@PostMapping(value = "/add")
	public Result<?> add(@RequestBody AddEventParam eventParam) {
		eventParam.setCreateTime(new Date());
		eventParam.setUpdateTime(new Date());
		Event event = eventService.saveMain(eventParam);
		//插入xxlJob
		iXxlJobInfoService.subjectInsert(event);
		return Result.OK("添加成功！");
	}
	
	/**
	 * 编辑
	 *
	 * @param addEventParam
	 * @return
	 */
	@ApiOperation(value="事件-编辑", notes="事件-编辑")
	@PutMapping(value = "/edit")
	public Result<?> edit(@RequestBody AddEventParam addEventParam) {
		Event byId = eventService.getById(addEventParam.getId());
		addEventParam.setUpdateTime(new Date());
		eventService.updateMain(addEventParam);
		//更新xxljob
		List<String> subjectCodes = new ArrayList<>();
		if(null!=addEventParam.getEventCode()){
			subjectCodes.add(addEventParam.getEventCode());
			iXxlJobInfoService.keyWordsUpdate(subjectCodes, String.valueOf(addEventParam.getStatus()));
		}else {
			return Result.error("eventCode 必传!");
		}


		//判断是否提取热词
		eventService.extractHotWords(addEventParam);
		if (!Objects.equals(byId.getStatus(), addEventParam.getStatus()) && addEventParam.getStatus() == 1){
			produceInfo.sendSubjectMsg(addEventParam.getEventCode());
		}
		return Result.OK("编辑成功!");
	}
	
	/**
	 * 通过id删除
	 *
	 * @param id
	 * @return
	 */
	@ApiOperation(value="事件-通过id删除", notes="事件-通过id删除")
	@DeleteMapping(value = "/delete")
	public Result<?> delete(@RequestParam(name="id",required=true) String id) {
		Event event = eventService.getById(id);
		iXxlJobInfoService.deleteByInfosourceCode(event.getEventCode());
		eventService.deleteMain(id);
		return Result.OK("删除成功!");
	}
	
	/**
	 * 批量删除
	 *
	 * @param ids
	 * @return
	 */
	@ApiOperation(value="事件-批量删除", notes="事件-批量删除")
	@DeleteMapping(value = "/deleteBatch")
	public Result<?> deleteBatch(@RequestParam(name="ids",required=true) String ids) {
		this.eventService.removeByIds(Arrays.asList(ids.split(",")));
		return Result.OK("批量删除成功！");
	}
	
	/**
	 * 通过id查询
	 *
	 * @param id
	 * @return
	 */
	@ApiOperation(value="事件-通过id查询", notes="事件-通过id查询")
	@GetMapping(value = "/queryById")
	public Result<?> queryById(@RequestParam(name="id",required=true) String id) {
		Event event = eventService.getById(id);
		return Result.OK(event);
	}


	 /**
	  * 专题信息源绑定
	  */
	 @PostMapping("/infoSourceBind")
	 public Object infoSourceBind(@RequestBody SubjectPage subjectPage) {
		 try {
			 JSONObject params =  ObjectUtil.objectToJSONObject(subjectPage);;
			 String url = SERVICE_PROJECT_URL + "event/infoSource/infoSourceBind";
			 return RestUtil.post(url, null, params);
		 } catch (Exception e) {
			 return null;
		 }
	 }
	 /**
	  * 查询信息源组的绑定列表
	  *
	  */
	 @GetMapping("/bindList")
	 public Object bindList(InfoSourceVo infoSourceVo,
							@RequestParam(name="ynBind") Integer ynBind,
							@RequestParam(name="groupId") String groupId,
							@RequestParam(name="pageNo", defaultValue="1") Integer pageNo,
							@RequestParam(name="pageSize", defaultValue="10") Integer pageSize) {
		 String url = SERVICE_PROJECT_URL + "event/infoSource/bindList";
		 JSONObject params = ObjectUtil.objectToJSONObject(infoSourceVo);
		 params.put("ynBind",ynBind);
		 params.put("groupId",groupId);
		 params.put("pageNo",pageNo);
		 params.put("pageSize",pageSize);
		 return RestUtil.get(url, params);
	 }
	 /**
	  * 关键词类别
	  *
	  */
	 @GetMapping("/keywordsType/rootListNoPage")
	 public Object keywordsList(@RequestParam(name = "contain", defaultValue = "false") Boolean contain,
								@RequestParam(name = "subjectId", defaultValue = "0") String subjectId) {
		 String url = SERVICE_PROJECT_URL + "event/keywordsType/rootListNoPage";
		 JSONObject params = new JSONObject();
		 params.put("contain",contain);
		 params.put("subjectId",subjectId);
		 return RestUtil.get(url, params);
	 }
	 /**
	  * 专题关键词绑定
	  */
	 @PostMapping("/keyWordsBind")
	 public Object keyWordsBind(@RequestBody SubjectPage subjectPage) {
		 try {
			 JSONObject params =  ObjectUtil.objectToJSONObject(subjectPage);;
			 String url = SERVICE_PROJECT_URL + "event/keyWordsBind";
			 return RestUtil.post(url, null, params);
		 } catch (Exception e) {
			 return null;
		 }
	 }
	 /**
	  * 专题关键词绑定
	  */
	 @PostMapping("/keyWordsEdit")
	 public Object keyWordsEdit(@RequestBody SubjectKeywordsMap subjectKeywordsMap) {
		 try {
			 JSONObject params =  ObjectUtil.objectToJSONObject(subjectKeywordsMap);;
			 String url = SERVICE_PROJECT_URL + "event/keyWords/edit";
			 return RestUtil.post(url, null, params);
		 } catch (Exception e) {
			 return null;
		 }
	 }

	/**
	 * 标签查询
	 */
	@PostMapping("/label/treeList")
	public Object labelTreeList(@RequestBody SubjectKeywordsMap subjectKeywordsMap) {
		try {
			JSONObject params =  ObjectUtil.objectToJSONObject(subjectKeywordsMap);;
			String url = SERVICE_PROJECT_URL + "event/label/treeList";
			return RestUtil.post(url, null, params);
		} catch (Exception e) {
			return null;
		}
	}



	 /**
	  * 传播路径
	  *
	  * @param eventId 事件id
	  */
	 @GetMapping("/propagationPath")
	 public Result propagationPath(@RequestParam String eventId) {
		 String key = Constants.SUBJECT_ANALYSIS_PRE + Constants.PROPAGATION_KEY + eventId;
		 PropagationPathVo pathVo = (PropagationPathVo) redisUtil.get(key);
		 if (ObjectUtils.isEmpty(pathVo)) {
			 pathVo = analysisService.propagationPath(eventId);
		 }
		 return Result.OK(pathVo);
	 }
	/**
	 * 事件脉络
	 *
	 * @param subjectId 专题id
	 * @param fakeNum 专题事件脉络展示 伪事件脉络 的资讯数量阈值
	 * @return com.zzsn.subjectAnalysis.common.Result
	 */
	@GetMapping("/eventContext")
	public Result eventContext(@RequestParam String subjectId,
							   @RequestParam(value = "fakeNum", required = false) Integer fakeNum) {
		if (fakeNum == null) {
			fakeNum = Constants.FAKE_NUM;
		}
		List<SubjectAnalysis> cacheList = analysisService.eventContext(subjectId, fakeNum);
		return Result.OK(cacheList);
	}

	/**
	 * 添上传
	 *
	 * @return
	 */
	@PostMapping(value = "/upload")
	@ResponseBody
	public Result<?> uploadKnowledge(HttpServletRequest request) {
		MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
		MultipartFile file = multipartRequest.getFile("file");// 获取上传文件对象
		String orgName = file.getOriginalFilename();// 获取文件名
		String url=eventService.upload(file);
		return Result.OK(url);
	}
}
