package com.zzsn.event.consumer;

import com.alibaba.fastjson2.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;

import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.Event;
import com.zzsn.event.entity.SubjectAnalysis;
import com.zzsn.event.service.EsService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.service.SubjectAnalysisService;
import com.zzsn.event.vo.SubjectDataVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;

/**
 * @author lkg
 * 监听消费kafka消息
 * 消费topic中消息(注意：如果监听的topic不存在则会自动创建)
 * @date 2022/7/15
 */
@Slf4j
@Component
public class KafkaConsumer {

    @Autowired
    private EsService esService;
    @Autowired
    private SubjectAnalysisService subjectAnalysisService;
    @Autowired
    private IEventService eventService;

    /**
     * 获取-观点分析-分析结果数据，并入库
     *
     * @param record 接受的kafka数据
     */
    @KafkaListener(topics = {Constants.VIEWPOINT_RECEIVE_TOPIC})
    public void viewPointAnalysis(ConsumerRecord<String, String> record) {
        String value = record.value();
        if (StringUtils.isNotEmpty(value)) {
            String subjectId = null;
            try {
                List<SubjectAnalysis> subjectAnalyses = JSON.parseArray(value, SubjectAnalysis.class);
                subjectAnalyses.forEach(e -> {
                    SubjectDataVo subjectDataVo = esService.queryInfo(e.getDataId());
                    Integer repeatNum = e.getRepeatNum();
                    if (repeatNum == 0) {
                        String repeatId = subjectDataVo.getRepeatId();
                        if (StringUtils.isNotEmpty(repeatId)) {
                            repeatNum = esService.getRepeatNum(repeatId);
                        }
                    }
                    if (repeatNum == 0) {
                        e.setRepeatNum(1);
                    } else {
                        e.setRepeatNum(repeatNum);
                    }
                    e.setTitle(removeNonBmpUniCodes(e.getTitle()));
                });
                Integer category = subjectAnalyses.get(0).getCategory();
                subjectId = subjectAnalyses.get(0).getSubjectId();
                Date analysisDate = subjectAnalyses.get(0).getAnalysisDate();
                LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
                queryWrapper.eq(SubjectAnalysis::getSubjectId, subjectId).eq(SubjectAnalysis::getCategory, category);
                subjectAnalysisService.remove(queryWrapper);
                subjectAnalysisService.saveBatch(subjectAnalyses);
                //更新专题最近一次观点分析的时间
                Event subject = new Event();
                subject.setId(subjectId);
                subject.setAnalysisTime(analysisDate);
                eventService.updateById(subject);
            } catch (Exception e) {
                e.printStackTrace();
            }
            log.info("id为-{}-的专题,此次-观点分析-数据更新完成", subjectId);
        }
    }

    /**
     * 获取-事件脉络-分析结果数据，并入库
     *
     * @param record 接受的kafka数据
     */
    @KafkaListener(topics = {Constants.EVENT_CONTEXT_RECEIVE_TOPIC})
    public void eventContext(ConsumerRecord<String, String> record) {
        String value = record.value();
        if (StringUtils.isNotEmpty(value)) {
            try {
                List<SubjectAnalysis> subjectAnalyses = JSON.parseArray(value, SubjectAnalysis.class);
                subjectAnalyses.forEach(e -> {
                    if (exist(e)) {
                        e.setTitle(removeNonBmpUniCodes(e.getTitle()));
                        subjectAnalysisService.save(e);
                    }
                });
                log.info("id为-{}-的专题,此次-事件脉络-数据更新完成", subjectAnalyses.get(0).getSubjectId());
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 获取-伪事件脉络-分析结果数据，并入库
     *
     * @param record 接受的kafka数据
     */
    @KafkaListener(topics = {Constants.FAKE_EVENT_CONTEXT_RECEIVE_TOPIC})
    public void eventContext_fake(ConsumerRecord<String, String> record) {
        String value = record.value();
        if (StringUtils.isNotEmpty(value)) {
            String subjectId = null;
            Integer category = 3;
            try {
                List<SubjectAnalysis> subjectAnalyses = JSON.parseArray(value, SubjectAnalysis.class);
                subjectId = subjectAnalyses.get(0).getSubjectId();
                subjectAnalyses.forEach(e -> {
                    e.setCategory(category);
                    e.setTitle(removeNonBmpUniCodes(e.getTitle()));
                });
                LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
                queryWrapper.eq(SubjectAnalysis::getSubjectId,subjectId).eq(SubjectAnalysis::getCategory,category);
                int count = subjectAnalysisService.count(queryWrapper);
                if (count>0) {
                    subjectAnalysisService.remove(queryWrapper);
                }
                subjectAnalysisService.saveBatch(subjectAnalyses);
            } catch (Exception e) {
                e.printStackTrace();
            }
            log.info("id为-{}-的专题,此次-伪事件脉络-数据更新完成", subjectId);
        }
    }

    private boolean exist(SubjectAnalysis subjectAnalyse){
        LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(SubjectAnalysis::getCategory,2).eq(SubjectAnalysis::getDataId,subjectAnalyse.getDataId())
                .eq(SubjectAnalysis::getSubjectId,subjectAnalyse.getSubjectId());
        SubjectAnalysis one = subjectAnalysisService.getOne(queryWrapper);
        return one == null;
    }

    //去除特殊的字符，例如表情符
    private String removeNonBmpUniCodes(String str) {
        return StringUtils.isEmpty(str) ? null : str.replaceAll("[^\\u0000-\\uFFFF]", "");
    }
}
