package com.zzsn.event.util;

import cn.hutool.core.lang.Snowflake;
import io.minio.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.net.URLDecoder;
import java.util.Date;

/**
 * minio文件上传工具类
 */
@Slf4j
public class MinioUtil {
    private static String minioUrl;
    private static String visitUrl;
    private static String minioName;
    private static String minioPass;
    private static String bucketName;

    public static void setMinioUrl(String minioUrl) {
        MinioUtil.minioUrl = minioUrl;
    }

    public static void setMinioName(String minioName) {
        MinioUtil.minioName = minioName;
    }

    public static void setMinioPass(String minioPass) {
        MinioUtil.minioPass = minioPass;
    }

    public static void setBucketName(String bucketName) {
        MinioUtil.bucketName = bucketName;
    }

    public static String getMinioUrl() {
        return minioUrl;
    }

    public static String getBucketName() {
        return bucketName;
    }

    public static String getVisitUrl() {
        return visitUrl;
    }

    public static void setVisitUrl(String visitUrl) {
        MinioUtil.visitUrl = visitUrl;
    }
    private static MinioClient minioClient = null;

    /**
     * 上传文件
     * @param file
     * @return
     */
    public static String upload(MultipartFile file, String bizPath, String customBucket) {
        String file_url = "";
        //update-begin-author:wangshuai date:20201012 for: 过滤上传文件夹名特殊字符，防止攻击
        bizPath=StrAttackFilter.filter(bizPath);
        //update-end-author:wangshuai date:20201012 for: 过滤上传文件夹名特殊字符，防止攻击
        String newBucket = bucketName;
        if(oConvertUtils.isNotEmpty(customBucket)){
            newBucket = customBucket;
        }
        try {
            initMinio(minioUrl, minioName,minioPass);
            // 检查存储桶是否已经存在
            if(minioClient.bucketExists(BucketExistsArgs.builder().bucket(newBucket).build())) {
                log.info("Bucket already exists.");
            } else {
                // 创建一个名为ota的存储桶
                minioClient.makeBucket(MakeBucketArgs.builder().bucket(newBucket).build());
                log.info("create a new bucket.");
            }
            InputStream stream = file.getInputStream();
            // 获取文件名
            String orgName = file.getOriginalFilename();
            if("".equals(orgName)){
                orgName=file.getName();
            }
            orgName = CommonUtils.getFileName(orgName);
            String objectName = bizPath+"/"+orgName.substring(0, orgName.lastIndexOf(".")) + "_" + System.currentTimeMillis() + orgName.substring(orgName.lastIndexOf("."));

            // 使用putObject上传一个本地文件到存储桶中。
            if(objectName.startsWith("/")){
                objectName = objectName.substring(1);
            }
            PutObjectArgs objectArgs = PutObjectArgs.builder().object(objectName)
                    .bucket(newBucket)
                    .contentType("application/octet-stream")
                    .stream(stream,stream.available(),-1).build();
            minioClient.putObject(objectArgs);
            stream.close();
            file_url = visitUrl+newBucket+"/"+objectName;
        }catch (Exception e){
            log.error(e.getMessage(), e);
        }
        return file_url;
    }

    /**
     * 文件上传
     * @param file
     * @param bizPath
     * @return
     */
    public static String upload(MultipartFile file, String bizPath) {
        return  upload(file,bizPath,null);
    }

    /**
     * 获取文件流
     * @param bucketName
     * @param objectName
     * @return
     */
    public static InputStream getMinioFile(String bucketName,String objectName){
        InputStream inputStream = null;
        try {
            initMinio(minioUrl, minioName, minioPass);
            GetObjectArgs objectArgs = GetObjectArgs.builder().object(objectName)
                    .bucket(bucketName).build();
            inputStream = minioClient.getObject(objectArgs);
        } catch (Exception e) {
            log.info("文件获取失败" + e.getMessage());
        }
        return inputStream;
    }

    /**
     * 删除文件
     * @param bucketName
     * @param objectName
     * @throws Exception
     */
    public static void removeObject(String bucketName, String objectName) {
        try {
            initMinio(minioUrl, minioName,minioPass);
            RemoveObjectArgs objectArgs = RemoveObjectArgs.builder().object(objectName)
                    .bucket(bucketName).build();
            minioClient.removeObject(objectArgs);
        }catch (Exception e){
            log.info("文件删除失败" + e.getMessage());
        }
    }

    /**
     * 获取文件外链
     * @param bucketName
     * @param objectName
     * @param expires
     * @return
     */
    public static String getObjectURL(String bucketName, String objectName, Integer expires) {
        initMinio(minioUrl, minioName,minioPass);
        try{
            GetPresignedObjectUrlArgs objectArgs = GetPresignedObjectUrlArgs.builder().object(objectName)
                    .bucket(bucketName)
                    .expiry(expires).build();
            String url = minioClient.getPresignedObjectUrl(objectArgs);
            return URLDecoder.decode(url,"UTF-8");
        }catch (Exception e){
            log.info("文件路径获取失败" + e.getMessage());
        }
        return null;
    }

    /**
     * 初始化客户端
     * @param minioUrl
     * @param minioName
     * @param minioPass
     * @return
     */
    private static MinioClient initMinio(String minioUrl, String minioName,String minioPass) {
        if (minioClient == null) {
            try {
                minioClient = MinioClient.builder()
                        .endpoint(minioUrl)
                        .credentials(minioName, minioPass)
                        .build();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return minioClient;
    }

    /**
     * 上传文件到minio
     * @param stream
     * @param relativePath
     * @return
     */
    public static String upload(InputStream stream,String relativePath) throws Exception {
        initMinio(minioUrl, minioName,minioPass);
        if(minioClient.bucketExists(BucketExistsArgs.builder().bucket(bucketName).build())) {
            log.info("Bucket already exists.");
        } else {
            // 创建一个名为ota的存储桶
            minioClient.makeBucket(MakeBucketArgs.builder().bucket(bucketName).build());
            log.info("create a new bucket.");
        }
        PutObjectArgs objectArgs = PutObjectArgs.builder().object(relativePath)
                .bucket(bucketName)
                .contentType("application/octet-stream")
                .stream(stream,stream.available(),-1).build();
        minioClient.putObject(objectArgs);
        stream.close();
        return visitUrl+bucketName+"/"+relativePath;
    }

    /**
     * 上传文件
     *
     * @param file         文件
     * @param contentType  类型，如果pdf文件或者图片，需要预览功能的话，必填.示例：pdf application/pdf；图片  image/png；image/jpeg
     * @param bizPath      文件夹名称
     * @param customBucket 自定义桶的名称
     * @author lkg
     * @date 2023/5/22
     */
    public static String upload(MultipartFile file, String contentType, String bizPath, String customBucket) {
        String fileUrl = null;
        try {
            if (StringUtils.isEmpty(bizPath)) {
                bizPath = "report";
            } else {
                //过滤上传文件夹名特殊字符，防止攻击
                bizPath = StrAttackFilter.filter(bizPath);
            }
            InputStream stream = file.getInputStream();
            // 获取文件名
            String orgName = file.getOriginalFilename();
            if (StringUtils.isEmpty(orgName)) {
                orgName = file.getName();
            }
            String objectName = bizPath + "/" + orgName;
            // 使用putObject上传一个本地文件到存储桶中。
            if (objectName.startsWith("/")) {
                objectName = objectName.substring(1);
            }
            fileUrl = uploadByStream(stream,contentType, customBucket,objectName);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return fileUrl;
    }

    /**
     * 通过文件流上传文件
     *
     * @param stream       文件流
     * @param contentType  类型，如果pdf文件或者图片，需要预览功能的话，必填.示例：pdf application/pdf；图片  image/png；image/jpeg
     * @param customBucket 自定义桶名称
     * @param relativePath 文件路径
     * @author lkg
     * @date 2023/5/22
     */
    public static String uploadByStream(InputStream stream, String contentType, String customBucket, String relativePath) throws Exception {
        initMinio(minioUrl, minioName, minioPass);
        if (StringUtils.isEmpty(customBucket)) {
            customBucket = bucketName;
        }
        if (!minioClient.bucketExists(BucketExistsArgs.builder().bucket(customBucket).build())) {
            minioClient.makeBucket(MakeBucketArgs.builder().bucket(customBucket).build());
        }
        if (StringUtils.isEmpty(contentType)) {
            contentType = "application/octet-stream";
        }
        int index = relativePath.lastIndexOf(".");
        relativePath = relativePath.substring(0,index) + "_" + System.currentTimeMillis() + relativePath.substring(index);
        if (relativePath.startsWith("/")) {
            relativePath = relativePath.substring(1);
        }
        PutObjectArgs objectArgs = PutObjectArgs.builder().object(relativePath)
                .bucket(customBucket)
                .contentType(contentType)
                .stream(stream, stream.available(), -1).build();
        minioClient.putObject(objectArgs);
        stream.close();
        return customBucket + "/" + relativePath;
    }

    /**
     * 通过字节上传文件
     *
     * @param bytes        字节
     * @param contentType  类型，如果pdf文件或者图片，需要预览功能的话，必填.示例：pdf application/pdf；图片  image/png；image/jpeg
     * @param customBucket 自定义桶名称
     * @param relativePath 文件路径
     * @author lkg
     * @date 2023/5/22
     */
    public static String uploadByByte(byte[] bytes, String contentType, String customBucket, String relativePath) throws Exception {
        ByteArrayInputStream stream = new ByteArrayInputStream(bytes);
        return uploadByStream(stream, contentType, customBucket, relativePath);
    }

    /**
     * 根据文件名生成文件存储路径(yyyy/yyyyMMdd/id)
     * @param fileName
     * @return
     */
    public static String getRelativePath(String fileName) {
        if(fileName == null || StringUtils.isEmpty(fileName) ){
            return null;
        }
        String suffix = fileName;
        if(fileName.lastIndexOf(".") > -1){
            suffix = fileName.substring(fileName.lastIndexOf(".") + 1);
        }
        String date = DateUtils.formatDate(new Date()).replaceAll("-", "");
        return date.substring(0,4)+"/"+date+"/"+ new Snowflake(1L,1L).nextIdStr()+"."+suffix;
    }

}
