package com.zzsn.event.controller;

import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.entity.LabelEntity;
import com.zzsn.event.entity.Report;
import com.zzsn.event.service.EsStatisticsService;
import com.zzsn.event.service.LabelEntityService;
import com.zzsn.event.service.ReportService;
import com.zzsn.event.util.CalculateUtil;
import com.zzsn.event.util.MinioUtil;
import com.zzsn.event.util.RestUtil;
import com.zzsn.event.vo.CountVO;
import com.zzsn.event.vo.MediaVO;
import com.zzsn.event.vo.Result;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;

import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletResponse;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

/**
 * 舆情分析
 *
 * @author lkg
 * @date 2024/1/24
 */
@RestController
@RequestMapping("/public/opinion/analysis")
public class EventAnalysisController {

    @Autowired
    private EsStatisticsService esStatisticsService;
    @Autowired
    private LabelEntityService labelEntityService;
    @Autowired
    private ReportService reportService;

    @Value("${subject.analysis-url:}")
    private String analysisUrl;



    /**
     * 下载报告
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/2/1
     */
    @GetMapping("/download")
    public void download(@RequestParam String subjectId,HttpServletResponse response){
        try {
            LambdaQueryWrapper<Report> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.select(Report::getReportName, Report::getFilePath)
                    .eq(Report::getSubjectId,subjectId);
            Report one = reportService.getOne(queryWrapper);
            String filePath = one.getFilePath();
            if (StringUtils.isNotEmpty(filePath)) {
                int index = filePath.indexOf("/");
                String bucketName = filePath.substring(0, index);
                String filename = filePath.substring(index);
                InputStream inputStream = MinioUtil.getMinioFile(bucketName, filename);
                byte[] bytes = IOUtils.toByteArray(inputStream);
                String fileName = one.getReportName() + ".docx";
                fileName = URLEncoder.encode(fileName, "UTF-8").replace("+", " ");
                response.setHeader("Access-Control-Expose-Headers", "Content-Disposition");
                response.setHeader("Content-Disposition", "attachment;filename=" + fileName);
                response.setContentLength(bytes.length);
                OutputStream out = response.getOutputStream();
                out.write(bytes);
                out.flush();
                out.close();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 总体分析
     *
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/1/24
     */
    @GetMapping("/total")
    public Result<?> totalAndMax(@RequestParam String subjectId, @RequestParam String startTime, @RequestParam String endTime) {
        Map<String, Object> map = esStatisticsService.totalAndMax(subjectId, null, null);
        long hours = DateUtil.between(DateUtil.parseDateTime(startTime), DateUtil.parseDateTime(endTime), DateUnit.HOUR);
        map.put("duration",hours);
        Object count = map.get("totalCount");
        String divide = CalculateUtil.divide(String.valueOf(count), String.valueOf(hours), 2);
        map.put("spread", divide);
        String mainReport = esStatisticsService.mainReport(subjectId);
        map.put("mainReport",mainReport);
        return Result.OK(map);
    }

    /**
     * 情感判断分析
     *
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param type      1-按小时;2-按天
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/orientation")
    public Result<?> orientation(@RequestParam String subjectId, @RequestParam String startTime,
                                 @RequestParam String endTime, @RequestParam Integer type) {
        List<CountVO> list = new ArrayList<>();
        String labelTypeId = "1631119596744265729";
        List<LabelEntity> labelEntities = labelEntityService.listByType(labelTypeId);
        AtomicLong total = new AtomicLong();
        labelEntities.forEach(e -> {
            CompletableFuture<CountVO> async = CompletableFuture.supplyAsync(() -> {
                CountVO countVO = esStatisticsService.orientation(subjectId, e.getId(), startTime, endTime, type);
                total.addAndGet(countVO.getValue());
                supply(countVO, startTime,endTime,type);
                return countVO;
            });
            try {
                CountVO countVO = async.get();
                list.add(countVO);
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        });
        for (CountVO countVO : list) {
            long value = countVO.getValue();
            long totalCount = total.get();
            String divide = CalculateUtil.divide(String.valueOf(value), String.valueOf(totalCount));
            String percentage = "0%";
            if (StringUtils.isNotEmpty(divide)) {
                percentage = CalculateUtil.percentage(Double.parseDouble(divide),false);
            }
            countVO.setPercentage(percentage);
        }
        return Result.OK(list);
    }


    /**
     * 影响力分析
     *
     * @param subjectId  专题id
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/influence")
    public Result<?> influence(@RequestParam String subjectId){
        List<CountVO> list = esStatisticsService.influence(subjectId, null, null);
        return Result.OK(list);
    }

    /**
     * 来源分析
     *
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param type      1-按小时;2-按天
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/source")
    public Result<?> source(@RequestParam String subjectId,@RequestParam String startTime,
                            @RequestParam String endTime, @RequestParam Integer type){
        List<CountVO> list = esStatisticsService.source(subjectId, startTime, endTime, type);
        for (CountVO countVO : list) {
            supply(countVO, startTime,endTime,type);
        }
        return Result.OK(list);
    }




    /**
     * 平台活跃度占比
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/origin")
    public Result<?> origin(@RequestParam String subjectId){
        List<CountVO> list = esStatisticsService.origin(subjectId, null, null);
        return Result.OK(list);
    }


    //补充缺失的时间
    private void supply(CountVO countVO, String startTime,String endTime,Integer type) {
        List<CountVO> list = new ArrayList<>();
        List<CountVO> children = countVO.getChildren();
        Map<String, CountVO> map = children.stream().collect(Collectors.toMap(CountVO::getName, item -> item, (k1, k2) -> k2));
        DateTime startDate = DateUtil.parseDateTime(startTime);
        DateTime endDate = DateUtil.parseDateTime(endTime);
        List<DateTime> rangeToList = new ArrayList<>();
        String format = null;
        if (type == 1) {
            rangeToList = DateUtil.rangeToList(startDate, endDate, DateField.HOUR_OF_DAY);
            format = "yyyy-MM-dd HH";
        } else if (type == 2) {
            rangeToList = DateUtil.rangeToList(startDate, endDate, DateField.DAY_OF_YEAR);
            format = "yyyy-MM-dd";
        }
        for (DateTime dateTime : rangeToList) {
            String date = DateUtil.format(dateTime,format);
            if (map.containsKey(date)) {
                list.add(countVO);
            } else {
                CountVO vo = new CountVO();
                vo.setName(date);
                vo.setValue(0L);
                list.add(vo);
            }
        }
        countVO.setChildren(list);
    }
}
